<?php

namespace EasyNolo\BancaSellaPro\Helper;

use Magento\Framework\Convert\DataObject;
use Magento\Quote\Api\Data\PaymentInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Payment;
use Magento\Framework\App\Helper\Context;
use Magento\Checkout\Model\Session;
use Magento\Store\Model\StoreManagerInterface;
use \Magento\Customer\Model\Session as CustomerSession;
use \EasyNolo\BancaSellaPro\Model\PaymentResponseFactory;
use \EasyNolo\BancaSellaPro\Model\Gestpay;
use EasyNolo\BancaSellaPro\Logger\Logger as CustomLogger;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{

    protected $_gestpay_logger = null;
    protected $_sessionFactory;
    protected $_custom_logger;

    public function __construct(
        Context $context,
        Session $checkoutSession,
        StoreManagerInterface $storeManager,
        PaymentResponseFactory $paymentResponseFactory,
        Gestpay $gestpay,
        CustomerSession $sessionFactory,
        CustomLogger $customLogger
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->storeManager = $storeManager;
        $this->paymentResponseFactory = $paymentResponseFactory;
        $this->gestpay = $gestpay;
        $this->_sessionFactory = $sessionFactory;
        $this->_custom_logger = $customLogger;
        parent::__construct($context);
    }

    public function log($msg)
    {
        if (is_array($msg)) {
            $msg = print_r($msg, true);
        }
        if ($this->scopeConfig->getValue('payment/easynolo_bancasellapro/log')) {
            // $this->_getLogger()->debug($msg);            
            $this->_custom_logger->debug($msg);
        }
    }


    function _getDecryptParams($a, $b)
    {
        $this->log('Imposto i parametri da inviare al decrypt');
        $params = array();
        $params['shopLogin'] = $a;
        $params['CryptedString'] = $b;
        if(!empty(trim($this->getApiKey()))) {
            $params['apikey'] = $this->getApiKey();
        }
        $this->log($params);
        return $params;
    }

    function getApiKey()
    {
        $api_key = $this->gestpay->getConfigData('api_key');
        return $api_key;
    }

    public function isElaborateS2S($order)
    {
        $state = $order->getState();
        if ($state == \Magento\Sales\Model\Order::STATE_NEW)
            return false;
        return true;
    }

    private function _getLogger()
    {
        if (!$this->_gestpay_logger) {
            $writer = new \Zend\Log\Writer\Stream(BP . '/var/log/EasyNolo_BancaSellaPro.log');
            $this->_gestpay_logger = new \Zend\Log\Logger();
            $this->_gestpay_logger->addWriter($writer);
        }
        return $this->_gestpay_logger;
    }

    function createUrl($url, $param)
    {
        $paramether = '';
        if (count($param)) {
            $paramether = '?';
            foreach ($param as $name => $value) {
                $paramether .= $name . '=' . $value . '&';
            }
        }
        return $url . $paramether;
    }

    function getGestPayJs($order)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $cryptDecrypt = $objectManager->create('EasyNolo\BancaSellaPro\Model\WS\CryptDecrypt');
        $gestpay = $order->getPayment()->getMethodInstance();
        $url = null;
        if ($gestpay->isIframeEnabled()) {
            $url = $gestpay->getIframeUrl();
        }
        return $url;
    }

    function getRedirectUrlToPayment($order)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $cryptDecrypt = $objectManager->create('EasyNolo\BancaSellaPro\Model\WS\CryptDecrypt');
        
        // $gestpay = $order->getPayment()->getMethodInstance();
        $params = $this->getGestpayParams($order, ['include_alternative_payments' => false]);
        try {
            $stringEncrypt = $cryptDecrypt->getEncryptString($params);
            $url = $this->gestpay->getRedirectPagePaymentUrl();
            return $url . '?a=' . $params['shopLogin'] . '&b=' . $stringEncrypt;
        } catch (\Exception $e) {
            $checkoutSession = $objectManager->create('Magento\Checkout\Model\Session');
            $checkoutSession->restoreQuote();
            $objectManager->create('\Magento\Framework\Message\ManagerInterface')->addError($e->getMessage());            
        }
        $urlBuilder = $objectManager->create('Magento\Framework\UrlInterface');
        return $urlBuilder->getUrl("checkout/cart/index");
    }

    private function _getBaseParams($order)
    {
        $gestpay = $order->getPayment()->getMethodInstance();

        $total = $gestpay->getTotalByOrder($order);

        $params = [];

        $params['shopLogin'] = $gestpay->getConfigData('merchant_id');
        $params['shopTransactionId'] = $order->getIncrementId();
        $params['uicCode'] = $gestpay->getConfigData('currency');
        if(!empty(trim($gestpay->getConfigData('api_key')))) {
            $params['apikey'] = $gestpay->getConfigData('api_key');
        }
        if ($gestpay->getConfigData('language')) {
            $params['languageId'] = $gestpay->getConfigData('language');
        }
        $params['amount'] = round($total, 2);

        if ($gestpay->getConfigData('tokenization')) {
            // create token for logged in
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $customerSession = $objectManager->create('\Magento\Customer\Model\Session');
            if($customerSession->isLoggedIn()) {
                if (!$gestpay->getConfigData('ask_tokenization') || (($additionalData = $gestpay->getInfoInstance()->getAdditionalInformation()) && !empty($additionalData['ask_tokenization']))) {
                    $params['requestToken'] = 'MASKEDPAN';
                }
            }
        }

        return $params;
    }

    protected function setPaymentParams(&$params, $order) {
        $method = $order->getPayment()->getMethodInstance();
        $additionalData = $method->getInfoInstance()->getAdditionalInformation();

        $allowLowRiskProfile = true;

        if (!empty($additionalData) && !empty($additionalData['alternative-payment'])) {
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $alternativeHelper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\AlternativePayments');
            $alternatives = $alternativeHelper->getAlternativePayments();
            if (!empty($alternatives) && !empty($alternatives[$additionalData['alternative-payment']])) {
                $method = $alternatives[$additionalData['alternative-payment']];

                if (isset($params['requestToken'])) {
                    unset($params['requestToken']);
                }

                $allowLowRiskProfile = false;

                $params['paymentTypes'] = array();
                $params['paymentTypes']['paymentType'] = array();
                $params['paymentTypes']['paymentType'][] = $method['type'];
                if (!empty($method['encrypt_helper'])) {
                    $helperPayment = $objectManager->create($method['encrypt_helper']);
                    if ($helperPayment) {
                        $additional = $helperPayment->getEncryptParams($order);
                        if ($additional && is_array($additional)) {
                            $params = array_merge_recursive($params, $additional);
                        }
                    }
                }
            }
        }

        // Low Risk Profile
        if (empty($params['requestToken']) && $allowLowRiskProfile) {
            if ($method->isLowRiskProfiledEnabled($order)) {
                $params['shopLogin'] = $method->getLowRiskProfileShopLogin();
            }
        }

        // Shop Login (for transactions with Token)
        if (!empty($params['tokenValue'])) {
            if ($tk_merchant_id = $method->getConfigData('tk_merchant_id')) {
                $params['shopLogin'] = $tk_merchant_id;
            }
        }
    }

    function getGestpayParams($order, $opts = [])
    {
        $params = $this->_getBaseParams($order);

        if (isset($opts['tokenValue'])) {
            unset($params['requestToken']);
            $params['tokenValue'] = $opts['tokenValue'];
        }

        $this->setPaymentParams($params, $order);

        $gestpay = $order->getPayment()->getMethodInstance();

        if ($gestpay->isRedEnabled()){
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $_redHelper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Red');
            $_redHelper->addParams($params, $order);
        }

        if ($gestpay->isRiskifiedEnabled()){
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $riskifiedHelper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Riskified');
            $riskifiedHelper->addParams($params, $order);
        }

        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $productMetadata  = $objectManager->create('Magento\Framework\App\ProductMetadataInterface');
        $moduleList  = $objectManager->create('Magento\Framework\Module\ModuleListInterface');
        $moduleVersion = $moduleList->getOne('EasyNolo_BancaSellaPro')['setup_version'];

        $params['customInfo'] = "Magento_Version=".$productMetadata->getVersion()."*P1*Module_Version=".$moduleVersion;

        $this->log($params);

        return $params;
    }

    public function getFastResultPayment($transactionResult)
    {
        if (!$transactionResult || $transactionResult == 'KO')
            return false;
        return true;
    }

    public function getFormattedToken($token)
    {
        return preg_replace("/([0-9]{2}).{10}([0-9]{4})/", "\${1}**********\${2}", $token);
    }

    public function getCardVendor($token)
    {
        if (preg_match("/^4[0-9]/", $token))
            return array('label' => 'Visa', 'id' => 'visa');
        elseif (preg_match("/^5[1-5]/", $token))
            return array('label' => 'MasterCard', 'id' => 'mastercard');
        elseif (preg_match("/^3[47]/", $token))
            return array('label' => 'Amex', 'id' => 'america-express');
        elseif (preg_match("/^3[068]/", $token))
            return array('label' => 'Diners Club', 'id' => 'diners');
        elseif (preg_match("/^6[05]/", $token))
            return array('label' => 'Discover', 'id' => 'discover');
        elseif (preg_match("/^21/", $token) || preg_match("/^18/", $token) || preg_match("/^35/", $token))
            return array('label' => 'JCB', 'id' => 'jcb');
        else
            return array('label' => 'unknown', 'id' => 'credit-card');
    }

    public function _checkEmailSend($order)
    {
        /**
         * we only want to send to customer about new order when there is no redirect to third party
         */
        if ($order && $order->getCanSendNewEmailFlag() && !$order->getSendEmail()) {
            try {
                $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
                $orderSender = $objectManager->create('Magento\Sales\Model\Order\Email\Sender\OrderSender');
                $orderSender->send($order);
            } catch (\Exception $e) {
                $this->_logger->critical($e);
            }
        }
    }


    public function isPaymentOk($a, $b)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $cryptDecrypt = $objectManager->create('EasyNolo\BancaSellaPro\Model\WS\CryptDecrypt');
        $registry = $objectManager->create('\Magento\Framework\Registry');

        $params = $this->_getDecryptParams($a, $b);
        $result = $cryptDecrypt->decryptRequest($params);
        if (!$result) return false;
        $orderId = $result->getShopTransactionID();

        $order = $objectManager->create('Magento\Sales\Model\Order')->load($orderId, 'increment_id');

        $registry->register('easynolo_bancasellapro_store_maked_order', $order->getStore());
        $registry->register('easynolo_bancasellapro_order', $order);

        if ($order->getId()) {
            if(!$this->checkoutSession->getLastRealOrderId()) {
                $this->checkoutSession->setLastRealOrderId($orderId);
                $this->log("getLastRealOrderId:". $this->checkoutSession->getLastRealOrderId());               
            }
            if(!$this->checkoutSession->getLastOrderId()) {
                $this->checkoutSession->setLastOrderId($order->getId()); 
                $this->log("getLastOrderId:". $this->checkoutSession->getLastOrderId());               
            }
            if(!$this->checkoutSession->getLastQuoteId()) {
                $this->checkoutSession->setLastQuoteId($order->getQuoteId());
                $this->log("getLastQuoteId:". $this->checkoutSession->getLastQuoteId());
            }
            $isCustomerloggedIn = $this->_sessionFactory->isLoggedIn();
            $this->log("getCustomerIsGuest:". $order->getCustomerIsGuest());
            if(!$isCustomerloggedIn && !$order->getCustomerIsGuest() && $order->getCustomerId()) {
                $customer_id = $order->getCustomerId();
                $this->_sessionFactory->loginById($customer_id);
                $this->log("loginById :". $customer_id);
            }

            $this->saveDecryptResponse($result); // To save decrypt response in payment_response table
            $payment = $order->getPayment();
            $payment->setAdditionalData(serialize($result->getData()));
            $method = $payment->getMethodInstance();

            switch ($result->getTransactionResult()) {

                case 'XX':

                    if(!$this->isElaborateS2S($order)){
                        $this->log('La transazione non è ancora stata inviata sul s2s');
                        $message = __("Authorizing amount of %1 is pending approval on gateway. GestPay Transaction ID: %2", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()), $result->getBankTransactionID());
                        if ($paymentMethod = $result->getPaymentMethod()) {
                            $message .= " (" . $paymentMethod . ")";
                        }
                        $order->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                        $status = $order->getConfig()->getStateDefaultStatus(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                        $order->setStatus($status);
                        $order->addStatusHistoryComment($message, $status);
                        $order->save();
                    }
                    
                    break;

                case 'OK':

                    if(!$this->isElaborateS2S($order)){
                        $riskified_skipped_payments = array('unionpay', 'alipay');
                        if ($this->isRedEnabled()):
                            switch ($result->getRedResponseCode()) {
                                case 'ACCEPT':
                                    $this->log('La tranzazione è gia stata salvata, non cambio lo stato');
                                    $message = __("Amount of %1 authorized.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                    // $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);
                                    // $status = $order->getConfig()->getStateDefaultStatus(\Magento\Sales\Model\Order::STATE_PROCESSING);

                                    $order->setState($method->getConfigData('order_status_ok_gestpay'));
                                    $status = $method->getConfigData('order_status_ok_gestpay');

                                    $order->setStatus($status);
                                    $order->addStatusHistoryComment($message, $status);
                                    //$order->setTotalPaid($order->getGrandTotal());
                                    $order->save();
    
                                    $this->_checkEmailSend($order);
                                    break;
    
                                default:
                                    $message = __("Authorization approved on gateway but RED return with '%1' status. GestPay Transaction ID: %2", $result->getRedResponseCode(), $result->getBankTransactionID());
                                    if ($paymentMethod = $result->getPaymentMethod()) {
                                        $message .= " (" . $paymentMethod . ")";
                                    }
                                    $payment->setAdditionalData(serialize($result->getData()))
                                        ->setTransactionAdditionalInfo(array(Payment\Transaction::RAW_DETAILS => $result->getData()), "");
                                    $payment->setTransactionId($result->getShopTransactionId())
                                        ->setCurrencyCode($order->getBaseCurrencyCode())
                                        ->setIsTransactionClosed(0);
    
                                    $status = Order::STATE_PAYMENT_REVIEW;
    
                                    if ($result->getRedResponseCode() == 'DENY')
                                        $status = $this->getRedConfigData('deny_order_status');
                                    elseif ($result->getRedResponseCode() == 'CHALLENGE')
                                        $status = $this->getRedConfigData('challenge_order_status');
                                    $order->setState(Order::STATE_PAYMENT_REVIEW);
                                    $order->setStatus($status);
                                    $order->addStatusHistoryComment($message, $status);
    
                                    $order->save();
                            }
                        elseif ($this->isRiskifiedEnabled() && !empty($result->getRiskResponseCode()) && !in_array(strtolower($result->getPaymentMethod()), $riskified_skipped_payments)):
                            switch ($result->getRiskResponseCode()) {
                                case 'approved':
                                    $this->log('La tranzazione è gia stata salvata, non cambio lo stato');
                                    $message = __("Amount of %1 authorized. GestPay Transaction ID: %2", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()), $result->getBankTransactionID());
                                    if ($paymentMethod = $result->getPaymentMethod()) {
                                        $message .= " (" . $paymentMethod . ")";
                                    }
                                    // $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);
                                    // $status = $order->getConfig()->getStateDefaultStatus(\Magento\Sales\Model\Order::STATE_PROCESSING);

                                    $order->setState($method->getConfigData('order_status_ok_gestpay'));
                                    $status = $method->getConfigData('order_status_ok_gestpay');

                                    $order->setStatus($status);
                                    $order->addStatusHistoryComment($message, $status);
    
                                    //$order->setTotalPaid($order->getGrandTotal());
                                    $order->save();
    
                                    $this->_checkEmailSend($order);
                                    break;
                                default:
                                    $message = __("Authorization approved on gateway but Riskified return with '%1' status. Response description: %2. GestPay Transaction ID: %3", $result->getRiskResponseCode(), $result->getRiskResponseDescription(), $result->getBankTransactionID());
                                    if ($paymentMethod = $result->getPaymentMethod()) {
                                        $message .= " (" . $paymentMethod . ")";
                                    }
                                    $payment->setAdditionalData(serialize($result->getData()))
                                        ->setTransactionAdditionalInfo(Payment\Transaction::RAW_DETAILS, $result->getData());
                                    $payment->setTransactionId($result->getShopTransactionId())
                                        ->setCurrencyCode($order->getBaseCurrencyCode())
                                        ->setIsTransactionClosed(0);
    
                                    $status = Order::STATE_PAYMENT_REVIEW;
    
                                    if ($result->getRiskResponseCode() == 'declined')
                                        $status = $this->getRiskifiedConfigData('declined_order_status');
                                    elseif ($result->getRiskResponseCode() == 'submitted')
                                        $status = $this->getRiskifiedConfigData('submitted_order_status');
                                    
                                    if($status == Order::STATE_CANCELED) {
                                        $order->cancel(); // To restore the canceled product quantity
                                    }
                                    $order->setState(Order::STATE_PAYMENT_REVIEW);
                                    $order->setStatus($status);
                                    $order->addStatusHistoryComment($message, $status);
    
                                    $order->save();
                                    if ($result->getRiskResponseCode() == 'declined') {
                                        $method->void($payment); //To perform callDeleteS2S, if riskified declined
                                    }
                            }
                        else:
                            $this->log('La tranzazione è gia stata salvata, non cambio lo stato');
                            $message = __("Amount of %1 authorized. GestPay Transaction ID: %2", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()), $result->getBankTransactionID());
                            if ($paymentMethod = $result->getPaymentMethod()) {
                                $message .= " (" . $paymentMethod . ")";
                            }
                            // $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);
                            // $status = $order->getConfig()->getStateDefaultStatus(\Magento\Sales\Model\Order::STATE_PROCESSING);

                            $order->setState($method->getConfigData('order_status_ok_gestpay'));
                            $status = $method->getConfigData('order_status_ok_gestpay');

                            $order->setStatus($status);
                            $order->addStatusHistoryComment($message, $status);
    
                            //$order->setTotalPaid($order->getGrandTotal());
                            $order->save();
    
                            $this->_checkEmailSend($order);
                        endif;
                    }
                    
                    break;

                case 'KO':
                    $this->log('Il web service ha restituito KO');
                    $message = __('Payment transaction not authorized: %1. GestPay Transaction ID: %2', $result->getErrorDescription(), $result->getBankTransactionID());
                    if ($paymentMethod = $result->getPaymentMethod()) {
                        $message .= " (" . $paymentMethod . ")";
                    }
                    $method = $order->getPayment()->getMethodInstance();
                    $order->cancel();
                    $order->setState($method->getConfigData('order_status_ko_gestpay'));
                    $status = $order->getConfig()->getStateDefaultStatus($method->getConfigData('order_status_ko_gestpay'));
                    $order->setStatus($status);
                    $order->addStatusHistoryComment($message, $status);
                    $order->save();
                    $message = __('Payment transaction not authorized: %1.', $result->getErrorDescription());
                    return [$message, $result];
            }
            if(!$this->checkoutSession->getLastSuccessQuoteId()) {
                $this->checkoutSession->setLastSuccessQuoteId($this->checkoutSession->getLastQuoteId());
            }
            return [true, $result];
        } else {
            $message = __("There was an error processing your order. Please contact us or try again later.");
            $this->log('L\'ordine restituito da bancasella non esiste. Increment id = ' . $orderId);
            return [$message, null];
        }
    }

    public function getRedConfigData($field, $storeId = null)
    {
        $path = 'payment/easynolo_bancasellapro_red/' . $field;
        return $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function isRedEnabled()
    {
        return $this->getRedConfigData('enable');
    }

    public function getRiskifiedConfigData($field, $storeId = null)
    {
        $path = 'payment/easynolo_bancasellapro_riskified/' . $field;
        return $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function isRiskifiedEnabled()
    {
        return $this->getRiskifiedConfigData('enable');
    }

    public function normalizePARes($pares)
    {
        $orderId = $this->checkoutSession->getLastRealOrderId();
        $this->log('PARes value for shopTransactionId: '.$orderId. ' - before normalization ('.$pares.')');
        $pares = trim(preg_replace("/\r|\n|\r\n/", "", trim($pares)));
        $pares = preg_replace('/[\xA\xD]/u', '', $pares);
        $this->log('PARes value for shopTransactionId: '.$orderId. ' - after normalization ('.$pares.')'); 
        return $pares;
    }

    public function normalize($string, $length=0)
    {
        $string = strip_tags(preg_replace("/&#?[a-z0-9]+;/i","",$string));
        $unwanted_strings = array("\r\n", "\r", "\n", "\"", "<", ">", "#", "%", "{", "}", "|", "\/", "\\", "^", "~", "[", "]", "'", "&");
        $filtered_string = trim(str_replace($unwanted_strings, '', $string));
        if ($length && strlen($filtered_string) > $length) {
            $filtered_string = substr($filtered_string, 0, $length);
        }
        return $filtered_string;
    }

    public function getStoreFromOrder()
    {      
        $order = $this->checkoutSession->getLastRealOrder();
        $this->storeManager->setCurrentStore($order->getStore()->getId());
        return true;
    }

    public function saveDecryptResponse($result)
    {
        $this->log('Save payment response');
        $response = $this->paymentResponseFactory->create();
        $response->setTransactionType($result->getTransactionType());
        $response->setTransactionResult($result->getTransactionResult());
        $response->setShopTransactionID($result->getShopTransactionID());
        $response->setBankTransactionID($result->getBankTransactionID());
        $response->setCurrency($result->getCurrency());
        $response->setAmount($result->getAmount());
        $response->setPaymentMethod($result->getPaymentMethod());
        $response->setToken($result->getToken());
        $response->setCreatedAt(time()); 
        $response->save();
    }
}