<?php

namespace EasyNolo\BancaSellaPro\Cron;

use Magento\Sales\Model\Order;

/**
 * Payment cron model
 *
 * Used for retrieving guaranteed payments status
 */
class CheckPendingReview {

    protected $salesOrderCollectionFactory = null;
    protected $helper = null;
    protected $s2s = null;
    protected $emulation = null;
    protected $storeManager;
    protected $scopeConfig;

    public function __construct(
        \Magento\Store\Model\App\Emulation $emulation,
        \EasyNolo\BancaSellaPro\Model\WS\WS2S $s2s,
        \EasyNolo\BancaSellaPro\Helper\Data $helper,
        \EasyNolo\BancaSellaPro\Model\Gestpay $gestpay,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $collectionFactory
    ) {
        $this->emulation = $emulation;
        $this->storeManager = $storeManager;
        $this->s2s = $s2s;
        $this->helper = $helper;
        $this->gestpay = $gestpay;
        $this->scopeConfig = $scopeConfig;
        $this->salesOrderCollectionFactory = $collectionFactory;
    }

    public function getSalesOrderCollection(array $filters = [])
    {
        /** @var \Magento\Sales\Model\ResourceModel\Order\Collection $salesOrderCollection */
        $salesOrderCollection = $this->salesOrderCollectionFactory->create();

        foreach ($filters as $field => $condition) {

            $salesOrderCollection->addFieldToFilter($field, $condition);
        }

        return $salesOrderCollection->load();
    }

    public function execute() {
        $this->helper->log('CRON execution started');
        $stores = $this->storeManager->getStores($withDefault = false);

        foreach ($stores as $store)
        {
            if($this->getConfigData('payment/easynolo_bancasellapro/use_s2s_api', $store->getStoreId())) {
                $this->emulation->startEnvironmentEmulation($store->getId());

                $statuses = array(
                    Order::STATE_PAYMENT_REVIEW,
                    Order::STATE_PENDING_PAYMENT,
                    Order::STATUS_FRAUD,
                    $this->getConfigData('order_status', $store->getStoreId())
                );
                if ($this->getConfigData('payment/easynolo_bancasellapro_riskified/enable', $store->getStoreId())) {
                    $status = $this->getConfigData('payment/easynolo_bancasellapro_riskified/submitted_order_status', $store->getStoreId());
                    if ($status) {
                        $statuses[$status] = $status;
                    }
                }
                if ($this->getConfigData('payment/easynolo_bancasellapro_red/enable', $store->getStoreId())) {
                    $status = $this->getConfigData('payment/easynolo_bancasellapro_red/challenge_order_status', $store->getStoreId());
                    if ($status) {
                        $statuses[$status] = $status;
                    }
                }

                if (!empty($statuses)) {
                    $collection = $this->getSalesOrderCollection([
                        'store_id' => $store->getId(),
                        'status' => $statuses
                    ]);

                    $collection->getSelect()->join(
                        ['payment' => $collection->getTable('sales_order_payment')],
                        'main_table.entity_id = payment.parent_id'
                    )->where(
                        'payment.method = \''.$this->gestpay::CODE.'\''
                    );
                    if(!empty($collection->getData())) {
                        $allowed_orders = array();
                        foreach($collection->getData() as $correct_orders) {
                            if($correct_orders['method'] == $this->gestpay::CODE) {
                                $allow_this = true;
                                if($correct_orders['status'] == $this->getConfigData('order_status', $store->getStoreId())) {
                                    $order_created_at = strtotime($correct_orders['created_at']);
                                    if($order_created_at > strtotime('-10 minutes')) {
                                        $allow_this = false;
                                    }
                                } 

                                if($allow_this) {
                                    array_push($allowed_orders, $correct_orders['increment_id']);                     
                                }
                            }
                        }
                        foreach ($collection as $order) {
                            if(in_array($order->getIncrementId(), $allowed_orders)) {
                                $paymentAdditionalData = unserialize($order->getPayment()->getAdditionalData());
                                $params = array();

                                $params['shopLogin'] = $this->getConfigData('payment/easynolo_bancasellapro/merchant_id', $store->getStoreId());
                                $params['shopTransactionId'] = $order->getIncrementId();
                                $params['bankTransactionId'] = isset($paymentAdditionalData['bank_transaction_i_d']) ? $paymentAdditionalData['bank_transaction_i_d'] : '';

                                $apikey = $this->getConfigData('payment/easynolo_bancasellapro/api_key', $store->getStoreId());
                                if(!empty(trim($apikey))) {
                                    $params['apikey'] = $apikey;
                                }

                                $result = $this->s2s->executeCallReadTrxS2S($params);
                                switch ($result->getTransactionResult()) {

                                    case 'OK':
                                        $riskified_skipped_payments = false;
                                        if (strpos(strtolower($result->getAuthorizationCode()), 's2p') !== false) {
                                            $riskified_skipped_payments = true;
                                        }
                                        if ($this->gestpay->isRedEnabled() && !empty($result->getRedResponseCode())):
                                            switch ($result->getRedResponseCode()) {
                                                case 'ACCEPT':
                                                    $state = Order::STATE_PROCESSING;
                                                    $message = __("Authorizing amount of %1 approved.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                                    $order->setState($state);
                                                    $order->addStatusHistoryComment($message, $this->getConfigData('order_status_ok_gestpay', $store->getStoreId()));
                                                    $order->save();
                                                    $this->helper->_checkEmailSend($order);
                                                    break;
                                                case 'DENY':
                                                    $message = __("Authorizing amount of %1 declined by RED.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                                    $order->cancel();

                                                    $order->addStatusHistoryComment($message, $this->getConfigData('payment/easynolo_bancasellapro_red/deny_order_status', $store->getStoreId()));
                                                    $order->save();
                                                    break;
                                            }
                                        elseif ($this->gestpay->isRiskifiedEnabled() && !empty($result->getRiskResponseCode()) && !$riskified_skipped_payments):
                                            switch ($result->getRiskResponseCode()) {
                                                case 'approved':
                                                    $state = Order::STATE_PROCESSING;
                                                    $message = __("Authorizing amount of %1 approved.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                                    $order->setState($state);
                                                    $order->addStatusHistoryComment($message, $this->getConfigData('order_status_ok_gestpay', $store->getStoreId()));
                                                    $order->save();
                                                    $this->helper->_checkEmailSend($order);
                                                    break;
                                                case 'declined':
                                                    $message = __("Authorizing amount of %1 declined by riskified.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                                    $order->setState(Order::STATE_PENDING_PAYMENT);
                                                    $order->setStatus(Order::STATE_PENDING_PAYMENT);
                                                    $order->save();
                                                    $order->cancel();

                                                    $order->addStatusHistoryComment($message, $this->getConfigData('payment/easynolo_bancasellapro_riskified/declined_order_status', $store->getStoreId()));
                                                    $order->save();
                                                    $this->gestpay->void($order->getPayment()); //To perform callDeleteS2S, if riskified declined
                                                    break;
                                            }
                                        else:
                                            $state = Order::STATE_PROCESSING;
                                            $message = __("Authorizing amount of %1 approved.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                            $order->setState($state);
                                            $order->addStatusHistoryComment($message, $this->getConfigData('order_status_ok_gestpay', $store->getStoreId()));
                                            $order->save();
                                            $this->helper->_checkEmailSend($order);
                                            break;
                                        endif;
                                        break;

                                    case 'KO':
                                        $message = __("Authorizing amount of %1 failed.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                                        $order->cancel();

                                        $order->addStatusHistoryComment($message, $this->getConfigData('order_status_ko_gestpay', $store->getStoreId()));
                                        $order->save();
                                        break;
                                }
                            }
                        }
                    }
                }

                // Stop store emulation process
                $this->emulation->stopEnvironmentEmulation();
            }
        }
        $this->helper->log('CRON execution ended');
        // return $this;
    }

    /**
     * Retrieve information from payment configuration
     *
     * @param string $field
     * @param int|string|null|\Magento\Store\Model\Store $storeId
     *
     * @return mixed
     */
    protected function getConfigData($field, $storeId)
    {
        if (strpos($field, '/') === false) {
            $path = 'payment/easynolo_bancasellapro/' . $field;
        } else {
            $path = $field;
        }
        return $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }
}