<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Drop\Tax\Plugin\Tax\Model\Sales\Total\Quote;

use Drop\Tax\Helper\ConfigurationHelper;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Tax\Api\Data\QuoteDetailsItemInterface;
use Magento\Tax\Api\Data\QuoteDetailsItemInterfaceFactory;

/**
 * Plugin for CommonTaxCollector to apply Tax Class ID from child item for configurable product
 */
class CommonTaxCollector
{
    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var ConfigurationHelper
     */
    private $configuration;

    /**
     * CommonTaxCollector constructor.
     * @param ScopeConfigInterface $scopeConfig
     * @param ProductRepositoryInterface $productRepository
     * @param StoreManagerInterface $storeManager
     * @param ConfigurationHelper $configurationHelper
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ProductRepositoryInterface $productRepository,
        StoreManagerInterface $storeManager,
        ConfigurationHelper $configurationHelper
    )
    {
        $this->scopeConfig = $scopeConfig;
        $this->productRepository = $productRepository;
        $this->storeManager = $storeManager;
        $this->configuration = $configurationHelper;
    }

    /**
     *
     * @param \Magento\Tax\Model\Sales\Total\Quote\CommonTaxCollector $subject
     * @param QuoteDetailsItemInterface $result
     * @param QuoteDetailsItemInterfaceFactory $itemDataObjectFactory
     * @param AbstractItem $item
     * @return QuoteDetailsItemInterface
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterMapItem(
        \Magento\Tax\Model\Sales\Total\Quote\CommonTaxCollector $subject,
        QuoteDetailsItemInterface $result,
        QuoteDetailsItemInterfaceFactory $itemDataObjectFactory,
        AbstractItem $item
    ): QuoteDetailsItemInterface
    {
        $website = $this->storeManager->getWebsite();

        if ($this->configuration->isEnabled($website)) {
            if ($item->getPrice() >= $this->configuration->getLimit($website)) {
                $product = $this->productRepository->getById($item->getProduct()->getId());
                $taxClassId = $this->configuration->useProductSecondaryTax($website) || !$product->getSecondaryTaxClassId() ?
                    $this->configuration->getSecondaryProductTaxClassId($website) : $product->getSecondaryTaxClassId();

                $result->getTaxClassKey()->setValue($taxClassId);
            }
        }

        return $result;
    }
}
