<?php
/**
 * @category    Magento 2
 * @copyright   Copyright (c) 2021 Drop (https://drop.it)
 * @author      Cavallaro <cavallaro@drop.it>
 */

namespace Drop\Tax\Block\Adminhtml\Rate;

use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Tax\Controller\RegistryConstants;

class Form extends \Magento\Tax\Block\Adminhtml\Rate\Form
{
    /**
     * Prepare form before rendering HTML.
     *
     * @return \Magento\Tax\Block\Adminhtml\Rate\Form
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     */
    protected function _prepareForm()
    {
        $taxRateId = $this->_coreRegistry->registry(RegistryConstants::CURRENT_TAX_RATE_ID);

        try {
            if ($taxRateId) {
                $taxRateDataObject = $this->_taxRateRepository->get($taxRateId);
            }
            // phpcs:ignore Magento2.CodeAnalysis.EmptyBlock
        } catch (NoSuchEntityException $e) {
            //tax rate not found//
        }

        $sessionFormValues = (array)$this->_coreRegistry->registry(RegistryConstants::CURRENT_TAX_RATE_FORM_DATA);
        $formData = isset($taxRateDataObject)
            ? $this->_taxRateConverter->createArrayFromServiceObject($taxRateDataObject)
            : [];
        $formData = array_merge($formData, $sessionFormValues);

        if (isset($formData['zip_is_range']) && $formData['zip_is_range'] && !isset($formData['tax_postcode'])) {
            $formData['tax_postcode'] = $formData['zip_from'] . '-' . $formData['zip_to'];
        }

        /** @var \Magento\Framework\Data\Form $form */
        $form = $this->_formFactory->create();

        $countries = $this->_country->toOptionArray(false, 'US');
        unset($countries[0]);

        if (!isset($formData['tax_country_id'])) {
            $formData['tax_country_id'] = $this->_scopeConfig->getValue(
                \Magento\Tax\Model\Config::CONFIG_XML_PATH_DEFAULT_COUNTRY,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        }

        if (!isset($formData['tax_region_id'])) {
            $formData['tax_region_id'] = $this->_scopeConfig->getValue(
                \Magento\Tax\Model\Config::CONFIG_XML_PATH_DEFAULT_REGION,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        }

        $regionCollection = $this->_regionFactory->create()->getCollection()->addCountryFilter(
            $formData['tax_country_id']
        );

        $regions = $regionCollection->toOptionArray();
        if ($regions) {
            $regions[0]['label'] = '*';
        } else {
            $regions = [['value' => '', 'label' => '*']];
        }

        $legend = $this->getShowLegend() ? __('Tax Rate Information') : '';
        $fieldset = $form->addFieldset(
            'base_fieldset',
            ['legend' => $legend, 'class' => 'admin__scope-old form-inline']
        );

        if (isset($formData['tax_calculation_rate_id']) && $formData['tax_calculation_rate_id'] > 0) {
            $fieldset->addField(
                'tax_calculation_rate_id',
                'hidden',
                ['name' => 'tax_calculation_rate_id', 'value' => $formData['tax_calculation_rate_id']]
            );
        }

        $fieldset->addField(
            'code',
            'text',
            [
                'name'     => 'code',
                'label'    => __('Tax Identifier'),
                'title'    => __('Tax Identifier'),
                'class'    => 'required-entry',
                'required' => true
            ]
        );

        $fieldset->addField(
            'zip_is_range',
            'checkbox',
            [
                'name'  => 'zip_is_range',
                'label' => __('Zip/Post is Range'),
                'value' => '1',
                'class' => 'zip-is-range-checkbox'
            ]
        );

        if (!isset($formData['tax_postcode'])) {
            $formData['tax_postcode'] = $this->_scopeConfig->getValue(
                \Magento\Tax\Model\Config::CONFIG_XML_PATH_DEFAULT_POSTCODE,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        }

        $fieldset->addField(
            'tax_postcode',
            'text',
            [
                'name'  => 'tax_postcode',
                'label' => __('Zip/Post Code'),
                'note'  => __(
                    "'*' - matches any; 'xyz*' - matches any that begins on 'xyz' and are not longer than %1.",
                    $this->_taxData->getPostCodeSubStringLength()
                ),
                'class' => 'validate-length maximum-length-' . $this->_taxData->getPostCodeSubStringLength()
            ]
        );

        $fieldset->addField(
            'zip_from',
            'text',
            [
                'name'      => 'zip_from',
                'label'     => __('Range From'),
                'required'  => true,
                'maxlength' => 9,
                'class'     => 'validate-digits',
                'css_class' => 'hidden'
            ]
        );

        $fieldset->addField(
            'zip_to',
            'text',
            [
                'name'      => 'zip_to',
                'label'     => __('Range To'),
                'required'  => true,
                'maxlength' => 9,
                'class'     => 'validate-digits',
                'css_class' => 'hidden'
            ]
        );

        $filterAmount = '0';
        if (array_key_exists('filter_amount', $formData)) {
            $filterAmount = ($formData['filter_amount'] ? '1' : '0');
        }
        $fieldset->addField(
            'filter_amount',
            'checkbox',
            [
                'name'    => 'filter_amount',
                'label'   => __('Use amount Limit'),
                'onclick' => 'this.value = this.checked ? 1 : 0;',
                'checked' => $filterAmount,
                'note'    => __('If not checked, amount below are not used')
            ],
            false,
            true
        );

        $fieldset->addField(
            'amount_from',
            'text',
            [
                'name'     => 'amount_from',
                'label'    => __('Amount From'),
                'class'    => 'validate-not-negative-number',
                'required' => false
            ],
            false,
            true
        );

        $fieldset->addField(
            'amount_to',
            'text',
            [
                'name'     => 'amount_to',
                'label'    => __('Amount To'),
                'class'    => 'validate-not-negative-number',
                'required' => false
            ],
            false,
            true
        );

        $fieldset->addField(
            'tax_region_id',
            'select',
            ['name' => 'tax_region_id', 'label' => __('State'), 'values' => $regions]
        );

        $fieldset->addField(
            'tax_country_id',
            'select',
            ['name' => 'tax_country_id', 'label' => __('Country'), 'required' => true, 'values' => $countries]
        );

        $fieldset->addField(
            'rate',
            'text',
            [
                'name'     => 'rate',
                'label'    => __('Rate Percent'),
                'title'    => __('Rate Percent'),
                'required' => true,
                'class'    => 'validate-not-negative-number'
            ]
        );

        $form->setAction($this->getUrl('tax/rate/save'));
        $form->setUseContainer(true);
        $form->setId(self::FORM_ELEMENT_ID);
        $form->setMethod('post');

        if (!$this->_storeManager->hasSingleStore()) {
            $form->addElement($this->_fieldsetFactory->create()->setLegend(__('Tax Titles')));
        }

        if (isset($formData['zip_is_range']) && $formData['zip_is_range']) {
            list($formData['zip_from'], $formData['zip_to']) = explode('-', $formData['tax_postcode']);
        }
        $form->setValues($formData);
        $this->setForm($form);

        $this->setChild(
            'form_after',
            $this->getLayout()->createBlock(
                \Magento\Framework\View\Element\Template::class
            )->setTemplate('Magento_Tax::rate/js.phtml')
        );

        return $this;
    }
}
