<?php
/**
 * @category    Magento 2
 * @copyright   Copyright (c) 2021 Drop (https://drop.it)
 * @author      Cavallaro <cavallaro@drop.it>
 */

namespace Drop\Tax\Model\Rate;

class CsvImportHandler extends \Magento\TaxImportExport\Model\Rate\CsvImportHandler
{
    /**
     * Retrieve a list of fields required for CSV file (order is important!)
     *
     * @return array
     */
    public function getRequiredCsvFields()
    {
        // indexes are specified for clarity, they are used during import
        return [
            0  => __('Code'),
            1  => __('Country'),
            2  => __('State'),
            3  => __('Zip/Post Code'),
            4  => __('Rate'),
            5  => __('Zip/Post is Range'),
            6  => __('Range From'),
            7  => __('Range To'),
            8  => __('Filter Amount'),
            9  => __('Amount From'),
            10 => __('Amount To')
        ];
    }

    /**
     * Import single rate
     *
     * @param array $rateData
     * @param array $regionsCache cache of regions of already used countries (is used to optimize performance)
     * @param array $storesCache cache of stores related to tax rate titles
     * @return array regions cache populated with regions related to country of imported tax rate
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function _importRate(array $rateData, array $regionsCache, array $storesCache)
    {
        // data with index 1 must represent country code
        $countryCode = $rateData[1];
        $country = $this->_countryFactory->create()->loadByCode($countryCode, 'iso2_code');
        if (!$country->getId()) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Country code is invalid: %1', $countryCode));
        }
        $regionsCache = $this->_addCountryRegionsToCache($countryCode, $regionsCache);

        // data with index 2 must represent region code
        $regionCode = $rateData[2];
        if (!empty($regionsCache[$countryCode][$regionCode])) {
            $regionId = $regionsCache[$countryCode][$regionCode] == '*' ? 0 : $regionsCache[$countryCode][$regionCode];
            // data with index 3 must represent postcode
            $postCode = empty($rateData[3]) ? null : $rateData[3];
            $modelData = [
                'code'           => $rateData[0],
                'tax_country_id' => $rateData[1],
                'tax_region_id'  => $regionId,
                'tax_postcode'   => $postCode,
                'rate'           => $rateData[4],
                'zip_is_range'   => $rateData[5],
                'zip_from'       => $rateData[6],
                'zip_to'         => $rateData[7],
                'filter_amount'  => $rateData[8],
                'amount_from'    => $rateData[9],
                'amount_to'      => $rateData[10]
            ];

            // try to load existing rate
            /** @var $rateModel \Magento\Tax\Model\Calculation\Rate */
            $rateModel = $this->_taxRateFactory->create()->loadByCode($modelData['code']);
            $rateModel->addData($modelData);

            // compose titles list
            $rateTitles = [];
            foreach ($storesCache as $fileFieldIndex => $storeId) {
                $rateTitles[$storeId] = $rateData[$fileFieldIndex];
            }

            $rateModel->setTitle($rateTitles);
            $rateModel->save();
        }

        return $regionsCache;
    }
}
