<?php


namespace Drop\SizeGuide\Console\Command\Import;


use Drop\SizeGuide\Api\WebsiteConversionRepositoryInterface;
use Drop\SizeGuide\Api\Data\WebsiteConversionInterfaceFactory;
use Drop\SizeGuide\Model\Data\WebsiteConversion as WebsiteConversionModel;
use Drop\SizeGuide\Helper\ConfigurationHelper;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Store\Api\WebsiteRepositoryInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class WebsiteConversion extends Command
{
    const FILENAME = 'filename';

    const MODULE_DIR = 'size_guide';

    protected $configuration;
    protected $appState;
    protected $directoryList;
    protected $file;
    protected $csv;
    protected $websiteRepository;
    protected $searchCriteriaBuilder;
    protected $websiteConversionRepository;
    protected $conversionInterfaceFactory;

    protected $generalCat;
    protected $generalSG;

    /**
     * WebsiteConversion constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param State $appState
     * @param DirectoryList $directoryList
     * @param File $file
     * @param Csv $csv
     * @param WebsiteRepositoryInterface $websiteRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param WebsiteConversionRepositoryInterface $websiteConversionRepository
     * @param WebsiteConversionInterfaceFactory $conversionInterfaceFactory
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        State $appState,
        DirectoryList $directoryList,
        File $file,
        Csv $csv,
        WebsiteRepositoryInterface $websiteRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        WebsiteConversionRepositoryInterface $websiteConversionRepository,
        WebsiteConversionInterfaceFactory $conversionInterfaceFactory
    )
    {
        $this->configuration = $configurationHelper;
        $this->appState = $appState;
        $this->directoryList = $directoryList;
        $this->file = $file;
        $this->csv = $csv;
        $this->websiteRepository = $websiteRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->websiteConversionRepository = $websiteConversionRepository;
        $this->conversionInterfaceFactory = $conversionInterfaceFactory;

        $this->generalCat = $this->configuration->getGeneralCategories();
        $this->generalSG = $this->configuration->getGeneralSizeGuide();

        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::FILENAME,
                null,
                InputOption::VALUE_REQUIRED,
                'get filename'
            ),
        ];
        $this->setName('drop:sg:import:conversion');
        $this->setDescription('Manual website conversion import');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     * @throws \Magento\Framework\Exception\FileSystemException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        if (!$filename = $input->getOption(self::FILENAME)) {
            $output->writeln("<error>field filename is required. The file must be uploaded to the var/" . self::MODULE_DIR . " folder</error>");
            return 0;
        }

        $path = $this->directoryList->getPath('var') . DS . self::MODULE_DIR . DS . $filename;
        if (!$this->file->isExists($path)) {
            $output->writeln("<error>file {$path} not exist</error>");
            return 0;
        }

        if (!$csvData = $this->readCsv($path)) {
            $output->writeln("<error>file is empty</error>");
            return 0;
        }

        $notSaved = 0;
        foreach ($csvData as $row) {
            if (!$this->manageRow($row)) {
                $notSaved++;
                continue;
            }
        }

        if ($notSaved > 0) {
            $output->writeln("<error>{$notSaved} rows not saved</error>");
        }

        return 1;
    }

    /**
     * @param $row
     * @return bool|void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function manageRow($row)
    {
        if (!$generalSizeId = $this->getGeneralColumnValue($row, $this->configuration->getGeneralSizeGuide(), $this->configuration->getGeneralSizeGuideColumn(), 'size_guide')) {
            throw new \Exception("general size not found");
            return;
        }

        if (!$categoryId = $this->getGeneralColumnValue($row, $this->configuration->getGeneralCategories(), $this->configuration->getGeneralCategoryColumn(), 'category')) {
            throw new \Exception("general category not found");
            return;
        }

        unset($row[$this->convertBoKeys($this->configuration->getGeneralSizeGuideColumn())]);
        unset($row[$this->convertBoKeys($this->configuration->getGeneralCategoryColumn())]);

        foreach ($row as $websiteCode => $value) {
            $websiteId = $this->websiteRepository->get($websiteCode)->getId();

            $searchCriteria = $this->searchCriteriaBuilder->addFilter(WebsiteConversionModel::GENERAL_SIZE_ID, $generalSizeId)
                ->addFilter(WebsiteConversionModel::WEBSITE_ID, $websiteId)
                ->addFilter(WebsiteConversionModel::CATEGORY_ID, $categoryId)
                ->create();
            $collection = $this->websiteConversionRepository->getList($searchCriteria)->getItems();
            if (count($collection) > 0) {
                // record già registrato
                return;
            }

            try {
                $websiteConversion = $this->conversionInterfaceFactory->create();
                $websiteConversion->setGeneralSizeId($generalSizeId);
                $websiteConversion->setWebsiteId($websiteId);
                $websiteConversion->setCategoryId($categoryId);
                $websiteConversion->setConversion($value);
                $this->websiteConversionRepository->save($websiteConversion);
            } catch (\Exception $e) {
                throw new \Exception("Save Website Conversion ERROR: {$e->getMessage()}");
                return;
            }
        }

        return true;
    }

    /**
     * @param $row
     * @param $generalValues
     * @param $generalColumn
     * @param $field
     * @return null
     */
    protected function getGeneralColumnValue($row, $generalValues, $generalColumn, $field)
    {
        foreach ($generalValues as $generalRow) {
            if (strtolower(trim($generalRow->$field) != strtolower(trim($row[$this->convertBoKeys($generalColumn)])))) {
                continue;
            }

            return $generalRow->id;
            break;
        }

        return null;
    }

    /**
     * @param $path
     * @return array
     * @throws \Exception
     */
    protected function readCsv($path): array
    {
        $data = $this->csv->getData($path);
        $header = [];
        $formatted = [];
        foreach ($data as $row => $values) {
            if ($row == 0) {
                $header = $this->convertToKey($values);
                continue;
            }

            foreach ($values as $key => $value) {
                $formatted[$row][$header[$key]] = $value;
            }
        }

        return $formatted;
    }

    /**
     * @param $data
     * @return mixed
     */
    protected function convertToKey($data)
    {
        foreach ($data as $key => $row) {
            $data[$key] = $this->convertBoKeys($row);
        }

        return $data;
    }

    /**
     * @param $value
     * @return string
     */
    protected function convertBoKeys($value)
    {
        return strtolower(trim(str_replace(" ", "_", $value)));
    }
}
