<?php

namespace Drop\SizeGuide\Console\Command\Import;

use Drop\SizeGuide\Api\SizeGuideRepositoryInterface;
use Drop\SizeGuide\Api\Data\SizeGuideInterfaceFactory;
use Drop\SizeGuide\Model\Data\SizeGuide as SizeGuideModel;
use Drop\SizeGuide\Api\AttributeValueRepositoryInterface;
use Drop\SizeGuide\Api\Data\AttributeValueInterfaceFactory;
use Drop\SizeGuide\Model\Data\AttributeValue as AttributeValueModel;
use Drop\SizeGuide\Helper\ConfigurationHelper;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Driver\File;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class SizeGuide extends Command
{
    const FILENAME = 'filename';

    const MODULE_DIR = 'size_guide';


    protected $configuration;
    protected $appState;
    protected $directoryList;
    protected $file;
    protected $csv;
    protected $productRepository;
    protected $searchCriteriaBuilder;
    protected $sizeGuideRepository;
    protected $sizeGuideInterfaceFactory;
    protected $attributeValueRepository;
    protected $attributeValueInterfaceFactory;

    protected $generalAttr;
    protected $generalCat;
    protected $generalSG;

    public function __construct(
        ConfigurationHelper $configurationHelper,
        State $appState,
        DirectoryList $directoryList,
        File $file,
        Csv $csv,
        ProductRepositoryInterface $productRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SizeGuideRepositoryInterface $sizeGuideRepository,
        SizeGuideInterfaceFactory $sizeGuideInterfaceFactory,
        AttributeValueRepositoryInterface $attributeValueRepository,
        AttributeValueInterfaceFactory $attributeValueInterfaceFactory
    )
    {
        $this->configuration = $configurationHelper;
        $this->appState = $appState;
        $this->directoryList = $directoryList;
        $this->file = $file;
        $this->csv = $csv;
        $this->productRepository = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sizeGuideInterfaceFactory = $sizeGuideInterfaceFactory;
        $this->sizeGuideRepository = $sizeGuideRepository;
        $this->attributeValueRepository = $attributeValueRepository;
        $this->attributeValueInterfaceFactory = $attributeValueInterfaceFactory;

        $this->generalAttr = $this->configuration->getGeneralAttributes();
        $this->generalCat = $this->configuration->getGeneralCategories();
        $this->generalSG = $this->configuration->getGeneralSizeGuide();

        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::FILENAME,
                null,
                InputOption::VALUE_REQUIRED,
                'get filename'
            ),
        ];
        $this->setName('drop:sg:import:guide');
        $this->setDescription('Manual size guide import');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     * @throws \Magento\Framework\Exception\FileSystemException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        if (!$filename = $input->getOption(self::FILENAME)) {
            $output->writeln("<error>field filename is required. The file must be uploaded to the var/" . self::MODULE_DIR . " folder</error>");
            return 0;
        }

        $path = $this->directoryList->getPath('var') . DS . self::MODULE_DIR . DS . $filename;
        if (!$this->file->isExists($path)) {
            $output->writeln("<error>file {$path} not exist</error>");
            return 0;
        }

        if (!$csvData = $this->readCsv($path)) {
            $output->writeln("<error>file is empty</error>");
            return 0;
        }

        $notSaved = 0;
        foreach ($csvData as $row) {
            if (!$this->manageRow($row)) {
                $notSaved++;
                continue;
            }
        }

        if ($notSaved > 0) {
            $output->writeln("<error>{$notSaved} rows not saved</error>");
        }

        return 1;
    }

    /**
     * @param $row
     * @return bool|void
     * @throws \Exception
     */
    protected function manageRow($row)
    {
        $this->searchCriteriaBuilder->addFilter('sku', $row[$this->convertBoKeys($this->configuration->getGeneralSkuColumn())]);
        $searchCriteria = $this->searchCriteriaBuilder->create();

        if (!$products = $this->productRepository->getList($searchCriteria)->getItems()) {
            return;
        }
        $product = reset($products);

        $categoryId = $this->getGeneralColumnValue($row, $this->configuration->getGeneralCategories(), $this->configuration->getGeneralCategoryColumn(), 'category');
        $generalSizeId = $this->getGeneralColumnValue($row, $this->configuration->getGeneralSizeGuide(), $this->configuration->getGeneralSizeGuideColumn(), 'size_guide');

        $searchCriteria = $this->searchCriteriaBuilder->addFilter(SizeGuideModel::PRODUCT_ID, $product->getEntityId())
            ->addFilter(SizeGuideModel::MASTER_ID, $generalSizeId)
            ->addFilter(SizeGuideModel::CATEGORY_ID, $categoryId)
            ->create();
        $collection = $this->sizeGuideRepository->getList($searchCriteria)->getItems();
        if (count($collection) > 0) {
            // record già registrato
            return;
        }

        try {
            $sizeGuide = $this->sizeGuideInterfaceFactory->create();
            $sizeGuide->setProductId($product->getEntityId());
            $sizeGuide->setMasterId($generalSizeId);
            $sizeGuide->setCategoryId($categoryId);
            $savedSizeGuide = $this->sizeGuideRepository->save($sizeGuide);
        } catch (\Exception $e) {
            throw new \Exception("Save SizeGuide ERROR: {$e->getMessage()}");
            return;
        }

        if (!$this->manageRowAttributes($savedSizeGuide->getEntityId(), $row)) {
            return;
        }

        return true;
    }

    /**
     * @param $sizeGuideId
     * @param $row
     * @return bool|void
     * @throws \Exception
     */
    protected function manageRowAttributes($sizeGuideId, $row)
    {
        foreach ($this->configuration->getGeneralAttributes() as $generalAttribute) {
            if (!isset($row[$generalAttribute->attribute]) || trim($row[$generalAttribute->attribute]) == '') {
                continue;
            }

            $searchCriteria = $this->searchCriteriaBuilder->addFilter(AttributeValueModel::SIZE_GUIDE_ID, $sizeGuideId)
                ->addFilter(AttributeValueModel::ATTRIBUTE_ID, $generalAttribute->id)
                ->create();
            $collection = $this->attributeValueRepository->getList($searchCriteria)->getItems();
            if (count($collection) > 0) {
                // record già registrato
                return;
            }

            try {
                $attributeValue = $this->attributeValueInterfaceFactory->create();
                $attributeValue->setSizeGuideId($sizeGuideId);
                $attributeValue->setAttributeId($generalAttribute->id);
                $attributeValue->setValue($row[$generalAttribute->attribute]);
                $savedSizeGuide = $this->attributeValueRepository->save($attributeValue);
            } catch (\Exception $e) {
                throw new \Exception("Save AttributeValue ERROR: {$e->getMessage()}");
            }
        }

        return true;
    }

    /**
     * @param $row
     * @param $generalValues
     * @param $generalColumn
     * @param $field
     * @return null
     */
    protected function getGeneralColumnValue($row, $generalValues, $generalColumn, $field)
    {
        foreach ($generalValues as $generalRow) {
            if (strtolower(trim($generalRow->$field) != strtolower(trim($row[$this->convertBoKeys($generalColumn)])))) {
                continue;
            }

            return $generalRow->id;
            break;
        }

        return null;
    }

    /**
     * @param $path
     * @return array
     * @throws \Exception
     */
    protected function readCsv($path): array
    {
        $data = $this->csv->getData($path);
        $header = [];
        $formatted = [];
        foreach ($data as $row => $values) {
            if ($row == 0) {
                $header = $this->convertToKey($values);
                continue;
            }

            foreach ($values as $key => $value) {
                $formatted[$row][$header[$key]] = $value;
            }
        }

        return $formatted;
    }

    /**
     * @param $data
     * @return mixed
     */
    protected function convertToKey($data)
    {
        foreach ($data as $key => $row) {
            $data[$key] = $this->convertBoKeys($row);
        }

        return $data;
    }

    /**
     * @param $value
     * @return string
     */
    protected function convertBoKeys($value)
    {
        return strtolower(trim(str_replace(" ", "_", $value)));
    }
}
