<?php

namespace Drop\SizeGuide\Block\Product;

use Drop\SizeGuide\Api\AttributeValueRepositoryInterface;
use Drop\SizeGuide\Model\Data\AttributeValue as AttributeValueModel;
use Drop\SizeGuide\Api\SizeGuideRepositoryInterface;
use Drop\SizeGuide\Model\Data\SizeGuide as SizeGuideModel;
use Drop\SizeGuide\Api\WebsiteConversionRepositoryInterface;
use Drop\SizeGuide\Model\Data\WebsiteConversion as WebsiteConversionModel;
use Drop\SizeGuide\Helper\ConfigurationHelper;
use Magento\Backend\Block\Template\Context;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Registry;
use Magento\Framework\View\Element\Template;
use Magento\Store\Model\StoreManagerInterface;

class SizeGuide extends Template
{
    const DEFAULT_IMG_ATTRIBUTE = "immagine";

    protected $searchCriteriaBuilder;
    protected $sizeGuideRepository;
    protected $attributeValueRepository;
    protected $websiteConversionRepository;
    protected $configuration;

    protected $generalAttr;
    protected $generalCat;
    protected $generalSG;
    protected $currentProduct;
    protected $currentWebsite;
    protected $img;

    /**
     * SizeGuide constructor.
     * @param Registry $registry
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SizeGuideRepositoryInterface $sizeGuideRepository
     * @param AttributeValueRepositoryInterface $attributeValueRepository
     * @param WebsiteConversionRepositoryInterface $websiteConversionRepository
     * @param ConfigurationHelper $configurationHelper
     * @param StoreManagerInterface $storeManager
     * @param Context $context
     * @param array $data
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function __construct(
        Registry $registry,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SizeGuideRepositoryInterface $sizeGuideRepository,
        AttributeValueRepositoryInterface $attributeValueRepository,
        WebsiteConversionRepositoryInterface $websiteConversionRepository,
        ConfigurationHelper $configurationHelper,
        StoreManagerInterface $storeManager,
        Context $context,
        array $data = []
    )
    {
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sizeGuideRepository = $sizeGuideRepository;
        $this->attributeValueRepository = $attributeValueRepository;
        $this->websiteConversionRepository = $websiteConversionRepository;
        $this->configuration = $configurationHelper;


        $this->generalAttr = $this->configuration->getGeneralAttributes();
        $this->generalCat = $this->configuration->getGeneralCategories();
        $this->generalSG = $this->configuration->getGeneralSizeGuide();
        $this->currentProduct = $registry->registry('current_product');
        $this->currentWebsite = $storeManager->getWebsite();

        parent::__construct($context, $data);
    }

    /**
     * @return array|false
     */
    public function getSizeGuides()
    {
        if (!$this->configuration->isEnabled()) {
            return null;
        }

        $simpleIds = $this->currentProduct->getTypeInstance()->getUsedProductIds($this->currentProduct);
        $searchCriteria = $this->searchCriteriaBuilder->addFilter(SizeGuideModel::PRODUCT_ID, $simpleIds, 'in')
            ->create();
        if (!$collection = $this->sizeGuideRepository->getList($searchCriteria)->getItems()) {
            return false;
        }

        $return = [];
        foreach ($collection as $sizeRow) {
            $cat = $this->getCategoryFromId($sizeRow);
            $size = $this->getGeneralSizeFromId($sizeRow);
            $attributes = $this->getAttributes($sizeRow);
            $conversion = $this->getConversion($size, $cat);
            $return[$cat->category][$size->size_guide] = $conversion + $attributes;
        }

        $return = $this->addTableHeader($return);

        return [
            'guides' => $return,
            'img'    => $this->img,
        ];
    }

    /**
     * @param $data
     * @return array
     */
    public function addTableHeader($data)
    {
        $formattedData = [];
        foreach ($data as $category => $sizes) {
            $formattedData[$category]['body'] = $sizes;
            foreach ($sizes as $attributes) {
                $formattedData[$category]['header'] = array_keys($attributes);
                array_unshift($formattedData[$category]['header'], $this->configuration->getGeneralSizeGuideLabel());
                break;
            }

        }

        return $formattedData;
    }

    /**
     * @param $sizeRow
     * @return array|false
     */
    protected function getAttributes($sizeRow)
    {
        $searchCriteria = $this->searchCriteriaBuilder->addFilter(AttributeValueModel::SIZE_GUIDE_ID, $sizeRow->getEntityId())
            ->create();
        if (!$collection = $this->attributeValueRepository->getList($searchCriteria)->getItems()) {
            return false;
        }

        $return = [];
        foreach ($collection as $item) {
            $generalAttr = $this->getAttributeFromId($item);
            // skippo l'attributo immagine perchè gestito a parte
            if ($generalAttr->attribute == self::DEFAULT_IMG_ATTRIBUTE) {
                $this->img = $item->getValue();
                continue;
            }

            $return[$generalAttr->attribute] = $item->getValue();
        }

        return $return;
    }

    /**
     * @param $generalSize
     * @param $generalCat
     * @return array
     */
    protected function getConversion($generalSize, $generalCat)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(WebsiteConversionModel::CATEGORY_ID, $generalCat->id)
            ->addFilter(WebsiteConversionModel::WEBSITE_ID, $this->currentWebsite->getWebsiteId())
            ->addFilter(WebsiteConversionModel::GENERAL_SIZE_ID, $generalSize->id)
            ->create();
        if (!$collection = $this->websiteConversionRepository->getList($searchCriteria)->getItems()) {
            return [];
        }

        $result = [];
        foreach ($collection as $item) {
            $result[$this->currentWebsite->getCode()] = $item->getConversion();
        }
        return $result;
    }

    /**
     * @param $sizeGuide
     * @return mixed
     */
    protected function getCategoryFromId($sizeGuide)
    {
        foreach ($this->generalCat as $generalCat) {
            if ($generalCat->id != $sizeGuide->getCategoryId()) {
                continue;
            }

            return $generalCat;
            break;
        }
    }

    /**
     * @param $attributeValue
     * @return mixed
     */
    protected function getAttributeFromId($attributeValue)
    {
        foreach ($this->generalAttr as $generalAttr) {
            if ($generalAttr->id != $attributeValue->getAttributeId()) {
                continue;
            }

            return $generalAttr;
            break;
        }
    }

    /**
     * @param $sizeGuide
     * @return mixed
     */
    protected function getGeneralSizeFromId($sizeGuide)
    {
        foreach ($this->generalSG as $generalSize) {
            if ($generalSize->id != $sizeGuide->getMasterId()) {
                continue;
            }

            return $generalSize;
            break;
        }
    }
}
