<?php

namespace Drop\Sendinblue\Model\Config\Structure\Element;

use \Magento\Config\Model\Config\Structure\Element\Group as OriginalGroup;

class Group
{

    const FIXED_FIELD_VALUE = 'fixed_value';
    const FIXED_FIELD_PREFIX = 'fixed_value_';
    const SENDINBLUE_FIELD_PREFIX = 'sendinblue_attribute_';

    private $newsletterFields = [];

    private $excludedCustomerFields = [
        'disable_auto_group_change',
        'vat_is_valid',
        'vat_request_id',
        'vat_request_date',
        'vat_request_success',
        'failures_num',
        'first_failure',
        'lock_expires',
        'region_id',

        //Always by subscriber
        'store_id',
//        'email',
//        'firstname',
//        'lastname',
//        'dob',
//        'gender',
        'ip_address',
//        'is_privacy_policy_accepted',
//        'is_profiling_accepted'
    ];
    private $excludedSubscriberFields = [];

    /**
     * Newsletter fields value to text conversion
     * For example subscriber_gender return 0 or 1. To return M or F is needed to load customer gender attribute and return option text.
     * @var array
     */
    private $subscriberCustomerFieldsMatches = [
        'subscriber_gender' => 'gender',
        'subscriber_dob' => 'dob'
    ];

    /**
     * @var \Drop\Sendinblue\Model\ResourceModel\Attributes\Collection
     */
    private $attributesCollection;
    /**
     * @var \Magento\Eav\Model\ResourceModel\Entity\Attribute\Collection
     */
    private $customerAttributesCollection;
    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    private $resource;
    /**
     * @var \Magento\Framework\App\DeploymentConfig
     */
    private $deploymentConfig;

    /**
     * Attributes constructor.
     * @param \Drop\Sendinblue\Model\ResourceModel\Attributes\Collection $attributesCollection
     * @param \Magento\Eav\Model\ResourceModel\Entity\Attribute\Collection $customerAttributesCollection
     * @param \Magento\Framework\App\ResourceConnection $resource
     * @param \Magento\Framework\App\DeploymentConfig $deploymentConfig
     */
    public function __construct(
        \Drop\Sendinblue\Model\ResourceModel\Attributes\Collection $attributesCollection,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute\Collection $customerAttributesCollection,
        \Magento\Framework\App\ResourceConnection $resource,
        \Magento\Framework\App\DeploymentConfig $deploymentConfig
    ) {
        $this->attributesCollection = $attributesCollection;
        $this->customerAttributesCollection = $customerAttributesCollection;
        $this->resource = $resource;
        $this->deploymentConfig = $deploymentConfig;
    }

    /**
     * @return string
     */
    protected function getDatabaseName() {
        return $this->deploymentConfig->get(
            \Magento\Framework\Config\ConfigOptionsListConstants::CONFIG_PATH_DB_CONNECTION_DEFAULT
            . '/' . \Magento\Framework\Config\ConfigOptionsListConstants::KEY_NAME
        );
    }

    /**
     * Get all newsletter_subscriber fields
     * Il sistema non è proprio bellissimo ma è veramente pratico per prendere tutti i campi della newsletter subscriber che non so come fare non essendo EAV
     * @return array
     */
    protected function getNewsletterSubscriberFields() {
        $tablePrefix = $this->deploymentConfig->get('db/table_prefix');
        $connection = $this->resource->getConnection(\Magento\Framework\App\ResourceConnection::DEFAULT_CONNECTION);
        $fields = $connection->fetchAll("SELECT COLUMN_NAME,COLUMN_COMMENT FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = '" . $this->getDatabaseName() . "' AND TABLE_NAME = '" . $connection->getTableName($tablePrefix . 'newsletter_subscriber') . "'");

        $newsletterSubscriberFields = [];
        foreach($fields as $field) {
            if(in_array($field['COLUMN_NAME'], $this->excludedSubscriberFields)) {
                continue;
            }

            $newsletterSubscriberFields[$field['COLUMN_NAME']] = 'Subscriber - ' . $field['COLUMN_COMMENT'];
        }

        return $newsletterSubscriberFields;
    }

    /**
     * Check if an attribute is of newsletter_suscriber table
     * @param $attributeCode
     * @return bool
     */
    public function isSubscriberAttribute($attributeCode) {
        if(empty($this->newsletterFields)) {
            $this->newsletterFields = $this->getNewsletterSubscriberFields();
        }
        if(array_key_exists($attributeCode, $this->newsletterFields)) {
            return true;
        }
        return false;
    }

    /**
     * Get all customer fields
     * @return array
     */
    protected function getCustomerFields() {
        $this->customerAttributesCollection->addFieldToFilter(\Magento\Eav\Model\Entity\Attribute\Set::KEY_ENTITY_TYPE_ID, [1, 2]);
        $attributes = $this->customerAttributesCollection->load()->getItems();

        $customerFields = [];
        foreach($attributes as $attribute)
        {
            if((empty($attribute->getFrontendLabel())) || in_array($attribute->getAttributeCode(), $this->excludedCustomerFields)) {
                continue;
            }

            $customerFields[$attribute->getAttributeCode()] = 'Customer - ' . $attribute->getFrontendLabel();
        }
        return $customerFields;
    }

    /**
     * Add all select option to admin field select
     * @return array
     */
    protected function getSendinblueAttributesOptions() {
        $fields = [];
        $fields[''] = '--Please Select--';
        $fields[self::FIXED_FIELD_VALUE] = 'Fixed Value';
        $fields = array_merge($fields, $this->getNewsletterSubscriberFields(), $this->getCustomerFields());

        $options = [];
        foreach($fields as $value => $label) {
            $options[$value] = [
                'value' => $value,
                'label' => $label
            ];
        }

        return $options;
    }

    /**
     * All Sendinblue dynamic attributes
     * @return array
     */
    private function getSendinblueAttributes() {
        $dynamicConfigFields = []; $i = 0;
        foreach($this->attributesCollection as $attribute) {
            $dynamicConfigFields[$attribute->getAttributeCode()] = [
                'id' => self::SENDINBLUE_FIELD_PREFIX . $attribute->getAttributeCode(),
                'type' => 'select',
                'sortOrder' => ($i * 10),
                'showInDefault' => '1',
                'showInWebsite' => '0',
                'showInStore' => '0',
                'frontend_class' => 'sendinblue-dynamic-config',
                'label' => $attribute->getFrontendLabel(),
                'options' => [
                    'option' => $this->getSendinblueAttributesOptions()
                ],
                '_elementType' => 'field',
                'path' => "newsletter/sendinblue"
            ];
            $i++;

            //Add hidden fixed field. Will be shown if fixed_value will be chosen
            $dynamicConfigFields[self::FIXED_FIELD_PREFIX . $attribute->getAttributeCode()] = [
                'id' => self::FIXED_FIELD_PREFIX . $attribute->getAttributeCode(),
                'type' => 'text',
                'sortOrder' => ($i * 10),
                'showInDefault' => '1',
                'showInWebsite' => '0',
                'showInStore' => '0',
                'frontend_class' => 'sendinblue-fixed-value',
                'label' => $attribute->getFrontendLabel() . ' fixed value',
                'depends' => [
                    'field' => [
                        self::SENDINBLUE_FIELD_PREFIX . $attribute->getAttributeCode() => ['value' => 'fixed_value']
                    ]
                ],
                '_elementType' => 'field',
                'path' => "newsletter/sendinblue"
            ];

            $i++;
        }

        return $dynamicConfigFields;
    }

    /**
     * Override setData when is sendinblue section to add dynamic fields
     * @param OriginalGroup $subject
     * @param callable $proceed
     * @param array $data
     * @param $scope
     * @return mixed
     */
    public function aroundSetData(OriginalGroup $subject, callable $proceed, array $data, $scope) {
        if(array_key_exists('path', $data) && array_key_exists('id', $data)) {
            if(($data['path'] == 'newsletter') && ($data['id'] == 'sendinblue')) {
                $dynamicFields = $this->getSendinblueAttributes();
                if(!empty($dynamicFields)) {
                    $data['children'] += $dynamicFields;
                }
            }
        }

        return $proceed($data, $scope);
    }

    /**
     * @return string
     */
    public function getSendinblueFieldPrefix() {
        return self::SENDINBLUE_FIELD_PREFIX;
    }

    /**
     * @return string
     */
    public function getFixedFieldPrefix() {
        return self::FIXED_FIELD_PREFIX;
    }

    /**
     * @return string
     */
    public function getFixedFieldValue() {
        return self::FIXED_FIELD_VALUE;
    }

    /**
     * Newsletter fields value to text conversion
     * For example subscriber_gender return 0 or 1. To return M or F is needed to load customer gender attribute and return option text.
     * @param $subscriberAttribute
     * @return mixed
     */
    public function getCustomerMatchField($subscriberAttribute) {
        if(empty($this->subscriberCustomerFieldsMatches[$subscriberAttribute])) {
            return $subscriberAttribute;
        }
        return $this->subscriberCustomerFieldsMatches[$subscriberAttribute];

    }

}
