<?php

namespace Drop\Sendinblue\Console\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Input\InputArgument;

class ExportTargetContact extends Command
{
    /**
     * @var \Drop\Sendinblue\Logger\Logger
     */
    protected $logger;
    /**
     * @var \Drop\Sendinblue\Model\Api\ContactsApi
     */
    private $contactsApi;
    /**
     * @var \Drop\Sendinblue\Model\SubscriberFactory $sendinblueSubscriberFactory;
    /**
     * @var \Magento\Framework\App\State
     */
    private $state;


    const ARGUMENT_CONTACT_EMAILS = 'emails';

    /**
     * ExportContacts constructor.
     * @param \Drop\Sendinblue\Model\SubscriberFactory $sendinblueSubscriberFactory
     * @param \Drop\Sendinblue\Model\Api\ContactsApi $contactsApi
     * @param \Drop\Sendinblue\Logger\Logger $logger
     * @param \Magento\Framework\App\State $state
     */
    public function __construct(
        \Drop\Sendinblue\Model\SubscriberFactory $sendinblueSubscriberFactory,
        \Drop\Sendinblue\Model\Api\ContactsApi $contactsApi,
        \Drop\Sendinblue\Logger\Logger $logger,
        \Magento\Framework\App\State $state
    ) {
        $this->sendinblueSubscriberFactory = $sendinblueSubscriberFactory;
        $this->contactsApi = $contactsApi;
        $this->logger = $logger;
        $this->state = $state;
        parent::__construct();
    }

    protected function configure()
    {
        //SAMPLE: m drop:sendinblue:export-target-contact c.pieroni@drop.it
        $this->setName('drop:sendinblue:export-target-contact')
            ->setDescription('Sendinblue export target contacts.');

        $this->addArgument(
            self::ARGUMENT_CONTACT_EMAILS,
            InputArgument::REQUIRED,
            'Contact emails (comma serated).'
        );
    }

    /**
     * Send last changed subscribers to Sendinblue
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute(InputInterface $input, OutputInterface $output) {
        try {
            // this tosses an error if the areacode is not set.
            $this->state->getAreaCode();
        } catch (\Exception $e) {
            $this->state->setAreaCode('adminhtml');
        }

        $subscribers = explode(',', $input->getArgument(self::ARGUMENT_CONTACT_EMAILS));

        $i = 0;
        foreach($subscribers as $subscriber) {
            if(!filter_var($subscriber, FILTER_VALIDATE_EMAIL)) {
                echo PHP_EOL . "EMAIL {$subscriber} was removed because is not valid.";
                unset($subscribers[$i]);
            }
            $i++;
        }

        $subscribersData = $this->sendinblueSubscriberFactory->create()->setTargetSubscribers($subscribers)->mapMagentoSubscriberFieldsToSendinblue();
        if(count($subscribersData) == 0) {
            $output->writeln('Contacts not found in Magento.');
            return $this;
        }

        $output->writeln('Found: ' . count($subscribersData) . ' contacts. Importing...');
        $this->contactsApi->setConnection()->importContacts($subscribersData);
        $output->writeln('Contacts exported to Sendinblue.');
        return $this;
    }

}
