<?php

namespace Drop\Sendinblue\Model;

use Drop\Sendinblue\Model\Config\Structure\Element\Group;
use Drop\Sendinblue\Model\ResourceModel\Attributes\Collection;

/**
 * Class Attributes
 *
 * @package \Drop\Sendinblue\Model
 */
class Subscriber extends \Magento\Framework\Model\AbstractModel {

    protected $subscribers;
    protected $unsubscribers;

    /**
     * @var \Magento\Newsletter\Model\ResourceModel\Subscriber\Collection
     */
    private $subscriberCollection;
    /**
     * @var Collection
     */
    private $attributeCollection;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;
    /**
     * @var \Drop\Sendinblue\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Sendinblue\Helper\Data
     */
    private $helper;
    /**
     * @var Config\Structure\Element\Group
     */
    private $group;
    /**
     * @var \Magento\Customer\Api\CustomerFactory
     */
    private $customerFactory;
    /**
     * @var \Magento\Eav\Model\Config
     */
    private $eavConfig;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * Export constructor.
     * @param \Magento\Newsletter\Model\ResourceModel\Subscriber\Collection $subscriberCollection
     * @param ResourceModel\Attributes\Collection $attributeCollection
     * @param \Magento\Eav\Model\Config $eavConfig
     * @param \Drop\Sendinblue\Helper\Data $helper
     * @param Group $group
     * @param \Magento\Customer\Model\CustomerFactory $customerFactory
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Drop\Sendinblue\Logger\Logger $logger
     */
    public function __construct(
        \Magento\Newsletter\Model\ResourceModel\Subscriber\Collection $subscriberCollection,
        Collection $attributeCollection,
        \Magento\Eav\Model\Config $eavConfig,
        \Drop\Sendinblue\Helper\Data $helper,
        Group $group,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Drop\Sendinblue\Logger\Logger $logger
    ) {
        $this->subscriberCollection = $subscriberCollection;
        $this->attributeCollection = $attributeCollection;
        $this->eavConfig = $eavConfig;
        $this->helper = $helper;
        $this->group = $group;
        $this->customerFactory = $customerFactory;
        $this->date = $date;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
    }

    /**
     * Last changed subscriber are subscriber that has field changes_status_at update at least one minute ago
     * @return $this
     */
    public function setLastChangedSubscriber() {
        $endDate = $this->date->gmtDate();
        $startDate = $this->date->date('Y-m-d H:i:s', $this->date->gmtTimestamp($endDate) - (60*120));

        $this->subscriberCollection->clear()->getSelect()->reset(\Zend_Db_Select::WHERE);
        $this->subscribers = $this->subscriberCollection
            ->addFieldToFilter('subscriber_status', ['eq' => 1])
            ->addFieldToFilter('change_status_at', ['gteq' => $startDate])
            ->addFieldToFilter('change_status_at', ['lteq' => $endDate]);

        if($this->helper->getDebug()) {
            $this->logger->info('Start date: ' . $startDate . ' - End date: ' . $endDate);
            $this->logger->info('Subscriber to export count: ' . $this->subscribers->count());
        }

        return $this;
    }

    public function setLastChangedUnsubscriber() {
        $endDate = $this->date->gmtDate();
        $startDate = $this->date->date('Y-m-d H:i:s', $this->date->gmtTimestamp($endDate) - (60*120));

        $this->subscriberCollection->clear()->getSelect()->reset(\Zend_Db_Select::WHERE); //Needed instead it will take the previous colelction (subscriber) XO
        $this->unsubscribers = $this->subscriberCollection
            ->addFieldToFilter('subscriber_status', ['neq' => 1])
            ->addFieldToFilter('change_status_at', ['gteq' => $startDate])
            ->addFieldToFilter('change_status_at', ['lteq' => $endDate]);

        if($this->helper->getDebug()) {
            $this->logger->info('Start date: ' . $startDate . ' - End date: ' . $endDate);
            $this->logger->info('Unsubscriber to export count: ' . $this->unsubscribers->count());
        }

        return $this;
    }

    public function setAllSubscriber() {
        $endDate = $this->date->gmtDate();

        $this->subscriberCollection->clear()->getSelect()->reset(\Zend_Db_Select::WHERE);
        $this->subscribers = $this->subscriberCollection
            ->addFieldToFilter('subscriber_status', ['eq' => 1])
            ->addFieldToFilter('change_status_at', ['lteq' => $endDate]);

        return $this;
    }

    public function setTargetSubscribers($subscribers) {
        if(empty($subscribers)) {
            return $this;
        }

        $this->subscribers = $this->subscriberCollection
            ->addFieldToFilter('subscriber_email', ['in' => $subscribers]);

        return $this;
    }

    public function getAllUnsubscriber() {
        $endDate = $this->date->gmtDate();

        $this->subscriberCollection->clear()->getSelect()->reset(\Zend_Db_Select::WHERE);
        $this->unsubscribers = $this->subscriberCollection
            ->addFieldToFilter('subscriber_status', ['neq' => 1])
            ->addFieldToFilter('change_status_at', ['lteq' => $endDate]);

        return $this->unsubscribers;
    }

    public function getTargetUnsubscribers($unsubscribers) {
        if(empty($unsubscribers)) {
            return $this;
        }

        $this->subscriberCollection->clear()->getSelect()->reset(\Zend_Db_Select::WHERE);
        $this->unsubscribers = $this->subscriberCollection
            ->addFieldToFilter('subscriber_status', ['neq' => 1])
            ->addFieldToFilter('subscriber_email', ['in' => $unsubscribers]);

        return $this;
    }

    /**
     * @return mixed
     */
    public function getLastChangedSubscriber() {
        if(empty($this->subscribers)) {
            $this->setLastChangedSubscriber();
        }
        return $this->subscribers;
    }

    /**
     * @return mixed
     */
    public function getLastChangedUnsubscriber() {
        if(empty($this->unsubscribers)) {
            $this->setLastChangedUnsubscriber();
        }
        return $this->unsubscribers;
    }

    /**
     * Logic
     * TODO: Non va bene la ripetizione fra subscriber e customer dei constrolli sulla tipologia, va fatto tanto refactoring.
     * @return array|bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function mapMagentoSubscriberFieldsToSendinblue() {
        if(!$this->subscribers->count()) {
            return false;
        }
        $mappingFields = $this->helper->getMappingFields();
        $listId = $this->helper->getListId();
        if(empty($listId)) {
            throw new \Exception('Cannot send contacts to Sendinblue because List Id field is empty.');
        }

        $contactData = []; $i = 0;
        foreach($this->subscribers as $subscriber) {

            $contactData[$i]['email'] = $subscriber->getSubscriberEmail();
            $contactData[$i]['listIds'] = [$listId];
            $contactData[$i]['updateEnabled'] = true;

            $customerId = $subscriber->getCustomerId();
            if(!empty($customerId)) {
                $customer = $this->customerFactory->create()->load($customerId);
                $customerAddresses = $customer->getAddresses();
            }

            foreach($mappingFields as $sendinblueField => $magentoField) {

                //FIXED FIELDS
                if($magentoField == $this->group->getFixedFieldValue()) {
                    $contactData[$i]['attributes'][$sendinblueField] = $this->helper->getFixedField($this->attributeCollection->getAttributeCode($sendinblueField)->getFirstItem()['attribute_code']);
                    continue;
                }

                //SUBSCRIBER FIELDS
                if($this->group->isSubscriberAttribute($magentoField)) {

                    if($magentoField == 'subscriber_status') {
                        $contactData[$i]['attributes'][$sendinblueField] = $this->helper->getTextNewsletterStatus($subscriber->getData($magentoField));
                        continue;
                    }

                    $attributeType = $this->eavConfig->getAttribute('customer', $this->group->getCustomerMatchField($magentoField))->getFrontendInput();

                    if(in_array($attributeType, ['select', 'multiselect'])) {
                        //Necessario matching fra attributo subscriber e attributo customer
                        if($magentoField == 'store_id') {
                            $stores = $this->storeManager->getStores(false, false);
                            foreach($stores as $store) {
                                if($store->getStoreId() == $subscriber->getData($magentoField)) {
                                    $contactData[$i]['attributes'][$sendinblueField] = $store['name'];
                                    continue;
                                }
                            }
                            continue;
                        }

                        $contactData[$i]['attributes'][$sendinblueField] = $subscriber->getData($magentoField);
                        continue;
                    }

                    if($attributeType == 'date') {
                        $contactData[$i]['attributes'][$sendinblueField] = $this->helper->getFormattedDate($subscriber->getData($magentoField));
                        continue;
                    }
                    if($attributeType == 'boolean') {
                        $contactData[$i]['attributes'][$sendinblueField] = $subscriber->getData($magentoField);
                        continue;
                    }
                    $contactData[$i]['attributes'][$sendinblueField] = $subscriber->getData($magentoField);
                    continue;
                }

                //CUSTOMER ADDRESS FIELDS
                if((!$customerId) || (!$customer) || (!$customerAddresses)) {
                    continue;
                }

                $dataFoundOnAddress = false;
                foreach($customerAddresses as $customerAddress) {
                    if(!empty($address = $customerAddress[$magentoField])) {
                        $contactData[$i]['attributes'][$sendinblueField] = $address;
                        $dataFoundOnAddress = true;
                        break;
                    }
                }
                if($dataFoundOnAddress) {
                    continue;
                }

                //CUSTOMER FIELDS
                $attributeType = $this->eavConfig->getAttribute('customer', $magentoField)->getFrontendInput();
                if(in_array($attributeType, ['select', 'multiselect'])) {
                    //Necessario matching fra attributo subscriber e attributo customer
                    $contactData[$i]['attributes'][$sendinblueField] = $this->getOptionText($customer, $magentoField, $customer->getData($magentoField));
                    continue;
                }
                if($attributeType == 'date') {
                    $contactData[$i]['attributes'][$sendinblueField] = $this->helper->getFormattedDate($customer->getData($magentoField));
                    continue;
                }
                if($attributeType == 'boolean') {
                    $contactData[$i]['attributes'][$sendinblueField] = $this->getBooleanText($customer, $magentoField, $customer->getData($magentoField));
                    continue;
                }
                $contactData[$i]['attributes'][$sendinblueField] = $customer->getData($magentoField);
            }

            $i++;
        }

        if($this->helper->getDebug()) {
            $this->logger->info(print_r($contactData, true));
        }

        return $contactData;
    }

    /**
     * Get Select/Multiselect option label
     * @param \Magento\Customer\Model\Customer $customer
     * @param $attributeCode
     * @param $attributeValue
     * @return mixed
     */
    public function getOptionText(\Magento\Customer\Model\Customer $customer, $attributeCode, $attributeValue) {
        return $customer->getResource()
            ->getAttribute($attributeCode)
            ->getSource()
            ->getOptionText($attributeValue);
    }

    /**
     * Get Boolean option label (Yes/No)
     * @param \Magento\Customer\Model\Customer $customer
     * @param $attributeCode
     * @param $attributeValue
     * @return mixed
     */
    public function getBooleanText(\Magento\Customer\Model\Customer $customer, $attributeCode, $attributeValue) {
        return $customer->getResource()
            ->getAttribute($attributeCode)
            ->getSource()
            ->getOptionText($attributeValue)
            ->getText();
    }

}
