<?php

namespace Drop\Sendinblue\Model\Api;

/**
 * Class Account
 *
 * @package \Drop\Sendinblue\Model
 */
class ContactsApi
{
    protected $connection;
    /**
     * @var \Drop\Sendinblue\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Sendinblue\Helper\Data
     */
    private $helper;
    /**
     * @var \Drop\Sendinblue\Model\Connection
     */
    private $connectionFactory;
    /**
     * @var \SendinBlue\Client\Api\ContactsApiFactory
     */
    private $contactsApiFactory;

    /**
     * AccountApi constructor.
     * @param \Drop\Sendinblue\Model\ConnectionFactory $connectionFactory
     * @param \SendinBlue\Client\Api\ContactsApiFactory $contactsApiFactory
     * @param \Drop\Sendinblue\Helper\Data $helper
     * @param \Drop\Sendinblue\Logger\Logger $logger
     */
    public function __construct(
        \Drop\Sendinblue\Model\ConnectionFactory $connectionFactory,
        \SendinBlue\Client\Api\ContactsApiFactory $contactsApiFactory,
        \Drop\Sendinblue\Helper\Data $helper,
        \Drop\Sendinblue\Logger\Logger $logger
    )
    {
        $this->connectionFactory = $connectionFactory;
        $this->contactsApiFactory = $contactsApiFactory;
        $this->helper = $helper;
        $this->logger = $logger;
    }

    public function setConnection() {
        if(!$this->connection) {
            $this->connection = $this->connectionFactory->create()->connect($this->contactsApiFactory);
        }
        return $this;
    }

    public function getConnection() {
        if(!$this->connection) {
            $this->setConnection();
        }
        return $this->connection;
    }

    /**
     * Import multiple contacts to Sendinblue
     * TODO: Utilizzare la funzione "importContacts" https://github.com/sendinblue/APIv3-php-library/blob/master/docs/Api/ContactsApi.md
     * @param $contacts
     */
    public function importContacts($contacts) {
        if(empty($contacts)) {
            return false;
        }

        foreach($contacts as $contact) {
            $this->importContact($contact);
        }
    }

    /**
     * Import Contact to Sendinblue
     * https://github.com/sendinblue/APIv3-php-library/blob/master/docs/Api/ContactsApi.md#createcontact
     * @param $contactData
     * @return mixed
     */
    public function importContact($contactData) {
        if(empty($contactData)) {
            return false;
        }

        try {
            $createContact = new \SendinBlue\Client\Model\CreateContact($contactData);
            return $this->connection->createContact($createContact);
        } catch (\Exception $e) {
            $this->logger->error('Exception when calling ContactsApi->createContact: ' . $e->getMessage());
        }
    }

    /**
     * Get contact information by email
     * https://github.com/sendinblue/APIv3-php-library/blob/master/docs/Api/ContactsApi.md#getcontactinfo
     * @param $email
     * @return mixed
     */
    public function getContactByEmail($email) {
        if(empty($email)) {
            return false;
        }

        try {
            return $this->connection->getContactInfo($email);
        } catch (\Exception $e) {
            $this->logger->error('Exception when calling ContactsApi->getContactInfo: ' . $e->getMessage());
        }
    }

    /**
     * List all attributes
     * https://github.com/sendinblue/APIv3-php-library/blob/master/docs/Api/AttributesApi.md#getattributes
     * @return mixed
     */
    public function getAttributes() {
        try {
            $result = $this->connection->getAttributes();
            return $result->getAttributes();
        } catch (\Exception $e) {
            $this->logger->error('Exception when calling ContactsApi->getAttributes: ' . $e->getMessage());
        }
    }

    /**
     * Delete contact by email
     * https://github.com/sendinblue/APIv3-php-library/blob/master/docs/Api/ContactsApi.md#deletecontact
     * @param $email
     * @return mixed
     */
    public function deleteContact($email) {
        if(empty($email)) {
            return false;
        }
        
        try {
            return $this->connection->deleteContact($email);
        } catch (\Exception $e) {
            $this->logger->error('Exception when calling ContactsApi->deleteContact: ' . $e->getMessage());
        }
    }

}
