<?php

namespace Drop\Sendinblue\Helper;

use Magento\Framework\App\Helper\Context;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{

    const XML_PATH_SENDINBLUE = 'newsletter/sendinblue';
    const XML_PATH_SENDINBLUE_ENABLED = 'newsletter/sendinblue/enabled';
    const XML_PATH_SENDINBLUE_API_KEY = 'newsletter/sendinblue/api_key';
    const XML_PATH_SENDINBLUE_LIST_ID = 'newsletter/sendinblue/list_id';
    const XML_PATH_SENDINBLUE_DATE_FORMAT = 'newsletter/sendinblue/date_format';
    const XML_PATH_SENDINBLUE_DEBUG = 'newsletter/sendinblue/debug';

    const DEFAULT_DATE_FORMAT = 'Y-m-d';

    private $newsletterStatusToText = [
        \Magento\Newsletter\Model\Subscriber::STATUS_SUBSCRIBED => 'Subscribed',
        \Magento\Newsletter\Model\Subscriber::STATUS_NOT_ACTIVE => 'Not Active',
        \Magento\Newsletter\Model\Subscriber::STATUS_UNSUBSCRIBED => 'Unsubscribed',
        \Magento\Newsletter\Model\Subscriber::STATUS_UNCONFIRMED => 'Unconfirmed'
    ];

    /**
     * @var \Drop\Sendinblue\Model\Config\Structure\Element\Group
     */
    private $group;
    /**
     * @var \Drop\Sendinblue\Model\ResourceModel\Attributes\CollectionFactory
     */
    private $attributesCollection;

    /**
     * Data constructor.
     * @param Context $context
     * @param \Drop\Sendinblue\Model\Config\Structure\Element\GroupFactory $group
     * @param \Drop\Sendinblue\Model\ResourceModel\Attributes\CollectionFactory $attributesCollection
     * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date
     */
    public function __construct(
        Context $context,
        \Drop\Sendinblue\Model\Config\Structure\Element\GroupFactory $group,
        \Drop\Sendinblue\Model\ResourceModel\Attributes\CollectionFactory $attributesCollection
    ) {
        $this->group = $group;
        $this->attributesCollection = $attributesCollection;
        parent::__construct($context);
    }

    /**
     * Retrieve scope config data
     *
     * @param $path
     * @param string $scopeType
     * @param null $scopeCode
     * @return string
     */
    public function getScopeConfig(
        $path,
        $scopeType = \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
        $scopeCode = null
    ) {
        return $this->scopeConfig->getValue($path, $scopeType, $scopeCode);
    }

    /**
     * @return string
     */
    public function isEnabled() {
        return $this->getScopeConfig(self::XML_PATH_SENDINBLUE_ENABLED);
    }

    /**
     * @return string
     */
    public function getApiKey() {
        return $this->getScopeConfig(self::XML_PATH_SENDINBLUE_API_KEY);
    }

    /**
     * @return string
     */
    public function getListId() {
        return (int) $this->getScopeConfig(self::XML_PATH_SENDINBLUE_LIST_ID);
    }

    /**
     * @return string
     */
    public function getDebug() {
        return (int) $this->getScopeConfig(self::XML_PATH_SENDINBLUE_DEBUG);
    }

    /**
     * @return string
     */
    public function getAllFields() {
        return $this->getScopeConfig(self::XML_PATH_SENDINBLUE);
    }

    /**
     * Get all setted fields and return an array that map sendinblue fields (key) to magento fields (value)
     * @return array
     */
    public function getMappingFields() {
        $allFields = $this->getAllFields();
        $fieldMapping = [];
        $group = $this->group->create();

        foreach($allFields as $fieldName => $fieldValue) {
            if(!empty($fieldValue)) {
                $attributesCollection = $this->attributesCollection->create();
                $frontendLabel = $attributesCollection->getAttributeFrontendLabel(str_replace($group->getSendinblueFieldPrefix(), '', $fieldName))->getFirstItem();
                if($frontendLabel['frontend_label']) {
                    $fieldName = $frontendLabel['frontend_label'];
                }
                $fieldMapping[$fieldName] = $fieldValue;
            }
        }
        return $fieldMapping;
    }

    /**
     * @param $fieldCode
     * @return string
     */
    public function getFixedField($fieldCode) {
        return $this->getScopeConfig(self::XML_PATH_SENDINBLUE . '/' . $this->group->create()->getFixedFieldPrefix() . $fieldCode);
    }

    /**
     * @param $string
     * @return string
     */
    public function getAttributeCode($string) {
        //Strip all characters but letters, numbers, and whitespace
        $string = preg_replace("/[^a-zA-Z0-9\s]/", "", $string);
        //Replace whitespace with dash
        return strtolower(str_replace([' '], '_', $string));
    }

    /**
     * Format date to admin format
     * @param $date
     * @return string
     */
    public function getFormattedDate($date) {
        $dateFormat = $this->getScopeConfig(self::XML_PATH_SENDINBLUE_DATE_FORMAT);
        if(empty($dateFormat)) {
            $dateFormat = self::DEFAULT_DATE_FORMAT;
        }
        $staticDate = new \DateTime();
        $staticDate->setTimestamp(strtotime($date));

        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $timezoneInterface = $objectManager->create('\Magento\Framework\Stdlib\DateTime\TimezoneInterface');
        return $timezoneInterface->date($staticDate)->format($dateFormat);
    }

    /**
     * Get subscriber status to text translation
     * @param $subscriberStatus
     * @return bool|string
     */
    public function getTextNewsletterStatus($subscriberStatus) {
        if(empty($this->newsletterStatusToText[$subscriberStatus])) {
            return $subscriberStatus;
        }
        return $this->newsletterStatusToText[$subscriberStatus];
    }

}
