<?php

namespace Drop\Sendinblue\Controller\Adminhtml\System\Config;

use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;

class Attributes extends Action
{

    /**
     * @var \Drop\Sendinblue\Model\Api\ContactsApi
     */
    private $contactsApi;
    /**
     * @var \Drop\Sendinblue\Model\AttributesFactory
     */
    private $attributesFactory;
    /**
     * @var \Drop\Sendinblue\Model\ResourceModel\Attributes\Collection
     */
    private $attributesCollection;
    /**
     * @var \Magento\Framework\Controller\Result\JsonFactory $resultJsonFactory
     */
    private $resultJsonFactory;
    /**
     * @var \Drop\Sendinblue\Logger\Logger
     */
    private $logger;
    /**
     * @var \Drop\Sendinblue\Helper\Data
     */
    private $helper;

    /**
     * @param Context $context
     * @param JsonFactory $resultJsonFactory
     * @param \Drop\Sendinblue\Model\Api\ContactsApi $contactsApi
     * @param \Drop\Sendinblue\Model\AttributesFactory $attributesFactory
     * @param \Drop\Sendinblue\Model\ResourceModel\Attributes\Collection $attributesCollection
     * @param \Drop\Sendinblue\Helper\Data $helper
     * @param \Drop\Sendinblue\Logger\Logger $logger
     */
    public function __construct(
        Context $context,
        \Drop\Sendinblue\Model\Api\ContactsApi $contactsApi,
        \Drop\Sendinblue\Model\AttributesFactory $attributesFactory,
        \Drop\Sendinblue\Model\ResourceModel\Attributes\Collection $attributesCollection,
        \Magento\Framework\Controller\Result\JsonFactory $resultJsonFactory,
        \Drop\Sendinblue\Helper\Data $helper,
        \Drop\Sendinblue\Logger\Logger $logger
    )
    {
        $this->contactsApi = $contactsApi;
        $this->attributesFactory = $attributesFactory;
        $this->attributesCollection = $attributesCollection;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->helper = $helper;
        $this->logger = $logger;
        parent::__construct($context);
    }

    /**
     * Download all new attributes, reset sendinblue_attributes table and re-add.
     * @return \Magento\Framework\App\ResponseInterface|\Magento\Framework\Controller\ResultInterface
     */
    public function execute()
    {
        /** @var \SendinBlue\Client\Model\Attributes $attributes */
        $attributes = $this->contactsApi->setConnection()->getAttributes();

        //Delete All Attributes
        if($deletedAttributes = $this->attributesCollection->deleteAll()) {
            $this->messageManager->addSuccess( $deletedAttributes . ' ' . __('Sendinblue attributes deleted'));
        }

        if(empty($attributes)) {
            $this->messageManager->addError( __('Cannot get attributes from Sendinblue'));
            return false;
        }

        $addedAttributes = 0;
        $errorAttributes = 0;
        foreach ($attributes as $attribute) {
            // Skip calculated attributes
            if($attribute->getCalculatedValue()) {
                continue;
            }

            $type = $attribute->getType();
            if(!empty($attribute->getEnumeration())) {
                $type = 'boolean';
            }
            if(empty($type)) {
                $this->logger->info('Empty type for attribute: ' . $attribute->getName());
                continue;
            }

            $model = $this->attributesFactory->create();
            $model->setAttributeCode($this->helper->getAttributeCode($attribute->getName()))
                ->setFrontendLabel($attribute->getName())
                ->setType($type);

            try {
                $model->save();
                $addedAttributes++;
            } catch (\Exception $e) {
                $this->logger->error(': ' . $e->getMessage());
                $errorAttributes++;
            }
        }

        $response['errors'] = false;
        $response['message'] = $addedAttributes . ' ' . __('Sendinblue attributes added!');
        if($addedAttributes) {
            $this->messageManager->addSuccess($response['message']);
        } elseif($errorAttributes) {
            $response['errors'] = true;
            $response['message'] = $errorAttributes . ' ' . __('Sendinblue attributes have errors. Check sendinblue.log for further information.');
            $this->messageManager->addErrorMessage($response['message']);
        }

        $response['redirectUrl'] = $this->_redirect->getRedirectUrl();
        $resultJson = $this->resultJsonFactory->create();
        return $resultJson->setData($response);
    }

//    protected function _isAllowed()
//    {
//          //TODO
//        return $this->_authorization->isAllowed('Drop_Sendinblue::config');
//    }
}
