<?php
/**
 * Author: Simone Monterubbiano <s.monterubbiano@drop.it>
 * Date: 04/03/2019
 * File name: Gsped.php
 * Project: module-rma-mirasvit
 */

namespace Drop\RmaMirasvit\Model\Consumer;

class Gsped implements \Rcason\Mq\Api\ConsumerInterface
{
    const STATUS_APPROVED = 2;
    const NUMERO_COLLI = 1;
    const FLAG_RESO = 1;

    /**
     * @var \Mirasvit\Rma\Model\RmaFactory
     */
    private $rmaFactory;
    /**
     * @var \Drop\Gsped\Api\Data\GspedInterface
     */
    private $gspedInterface;
    /**
     * @var \Drop\Gsped\Helper\Data
     */
    private $gspedHelper;
    /**
     * @var \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface
     */
    private $rmaManagement;
    /**
     * @var \Magento\Directory\Model\RegionFactory
     */
    private $regionFactory;
    /**
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    private $directoryList;
    /**
     * @var \Mirasvit\Rma\Api\Repository\AttachmentRepositoryInterface
     */
    private $attachmentRepository;
    /**
     * @var \Drop\RmaMirasvit\Helper\Data
     */
    private $helper;
    /**
     * @var \Mirasvit\Rma\Helper\Rma\Event
     */
    private $rmaEvent;
    /**
     * @var \Mirasvit\Rma\Model\AttachmentFactory
     */
    private $attachmentFactory;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    private $date;

    /**
     * Gsped constructor.
     * @param \Mirasvit\Rma\Model\RmaFactory $rmaFactory
     * @param \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement
     * @param \Mirasvit\Rma\Api\Repository\AttachmentRepositoryInterface $attachmentRepository
     * @param \Mirasvit\Rma\Helper\Rma\Event $rmaEvent
     * @param \Mirasvit\Rma\Model\AttachmentFactory $attachmentFactory
     * @param \Drop\Gsped\Api\Data\GspedInterface $gspedInterface
     * @param \Drop\Gsped\Helper\Data $gspedHelper
     * @param \Drop\RmaMirasvit\Helper\Data $helper
     * @param \Magento\Directory\Model\RegionFactory $regionFactory
     * @param \Magento\Framework\Filesystem\DirectoryList $directoryList
     */
    public function __construct(
        \Mirasvit\Rma\Model\RmaFactory $rmaFactory,
        \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement,
        \Mirasvit\Rma\Api\Repository\AttachmentRepositoryInterface $attachmentRepository,
        \Mirasvit\Rma\Helper\Rma\Event $rmaEvent,
        \Mirasvit\Rma\Model\AttachmentFactory $attachmentFactory,
        \Drop\Gsped\Api\Data\GspedInterface $gspedInterface,
        \Drop\Gsped\Helper\Data $gspedHelper,
        \Drop\RmaMirasvit\Helper\Data $helper,
        \Magento\Directory\Model\RegionFactory $regionFactory,
        \Magento\Framework\Filesystem\DirectoryList $directoryList
    )
    {
        $this->rmaFactory = $rmaFactory;
        $this->gspedInterface = $gspedInterface;
        $this->gspedHelper = $gspedHelper;
        $this->rmaManagement = $rmaManagement;
        $this->regionFactory = $regionFactory;
        $this->directoryList = $directoryList;
        $this->attachmentRepository = $attachmentRepository;
        $this->attachmentFactory = $attachmentFactory;
        $this->helper = $helper;
        $this->rmaEvent = $rmaEvent;
    }

    public function process($rmaId)
    {
        if (empty($rmaId)) {
            throw new \Exception("rmaId is empty");
        }

        try {
            // Load rma by id
            $rma = $this->rmaFactory->create();
            $rma->load($rmaId);
        } catch (\Exception $e) {
            throw new \Exception("Cannot load RMA by id - " . $e->getMessage());
        }

        try {
            // Load order by rma
            $order = $this->rmaManagement->getOrder($rma);
        } catch (\Exception $e) {
            throw new \Exception("Cannot load order from RMA - " . $e->getMessage());
        }

        // Prepare base data for Gsped
        $data = $this->prepareGspedData($order);

        // Merge carrier code/service
        $carrier = $this->getCarrier($order);
        $data = array_merge($data, $carrier);

        // Merge receipt gsped address
        $rcpt = $this->helper->getGspedRcpt();
        $data = array_merge($data, $rcpt);

        // Call Gsped to create shipment
        $result = $this->gspedInterface->createShipment($data);

        // Get shipment label created
        if (!isset($result->label_path[0])) {
            throw new \Exception("No label found un Gsped response");
        }
        $labelPath = $result->label_path[0];
        $explode = explode("/", $labelPath);
        $fileName = end($explode);

        try {
            // Delete old attachment from database if it exists
            $this->deleteAttachmentIfExist($rmaId);

            // Save attachment to db table
            $attachment = $this->attachmentRepository->create();
            $body = @file_get_contents(addslashes($labelPath));
            $attachment
                ->setItemType("return_label")
                ->setItemId($rmaId)
                ->setName($fileName)
                ->setBody($body)
                ->setSize(filesize($labelPath))
                ->setType("application/pdf")
                ->save();
        } catch (\Exception $e) {
            throw new \Exception("Cannot save label attachment - " . $e->getMessage());
        }

        try {
            // Set tracking link and status to "Approved"
            $rma
                ->setTrackingLink($result->tracking_link)
                ->setStatusId(self::STATUS_APPROVED)
                ->save();
        } catch (\Exception $e) {
            throw new \Exception("Cannot set tracking or status on RMA - " . $e->getMessage());
        }

        // Call method for sending email
        $this->rmaEvent->onRmaStatusChange($rma);
    }

    /**
     * @param $order
     * @return array
     */
    private function prepareGspedData($order)
    {
        // Load shipping region
        $shippingRegion = $this->regionFactory->create()->load($order->getShippingAddress()->getRegionId());

        switch ($order->getShippingAddress()->getCountryId()):
            case 'IT';
                $senderProv = $shippingRegion->getCode();
                break;
            case 'US';
            case 'CA';
                $senderProv = substr($shippingRegion->getCode(), 0, 2);
                break;
            default;
                $senderProv = '';
                break;
        endswitch;

        $data = [
            "client_id"           => $this->gspedHelper->getRmaClientId(),
            "colli"               => self::NUMERO_COLLI,
            "flag_reso"           => self::FLAG_RESO,
            "ddt_alpha"           => (string)$order->getIncrementId() . '-R',
            "ddt_num"             => (string)$order->getIncrementId(),
            "peso"                => round($order->getWeight(), 2),
            "sender_name"         => $order->getShippingAddress()->getFirstname() . " " . $order->getShippingAddress()->getLastname(),
            "sender_addr"         => $order->getShippingAddress()->getStreet()[0],
            "sender_cap"          => $order->getShippingAddress()->getPostcode(),
            "sender_city"         => $order->getShippingAddress()->getCity(),
            "sender_contact"      => $order->getShippingAddress()->getFirstname() . " " . $order->getShippingAddress()->getLastname(),
            "sender_prov"         => $senderProv,
            "sender_country_code" => $order->getShippingAddress()->getCountryId(),
            "sender_email"        => $order->getShippingAddress()->getEmail(),
            "sender_phone"        => $order->getShippingAddress()->getTelephone()
        ];

        return $data;
    }

    /** Get carrier code/service based on shipment countryId
     * @param $order
     * @return array
     */
    private function getCarrier($order)
    {
        $countryId = $order->getShippingAddress()->getCountryId();
        $carrier = [
            "corriere" => ($countryId == "IT") ? $this->gspedHelper->getRmaCarrierCode() : $this->gspedHelper->getInternationalRmaCarrierCode(),
            "servizio" => ($countryId == "IT") ? $this->gspedHelper->getRmaCarrierService() : $this->gspedHelper->getInternationalRmaCarrierService()
        ];
        return $carrier;
    }

    /**
     * Delete RMA attachment from database if exist
     * @param $rmaId
     * @throws \Exception
     */
    private function deleteAttachmentIfExist($rmaId)
    {
        try {
            $attachmentCollection = $this->attachmentFactory->create()->getCollection();
            $attachmentCollection->addFieldToFilter("item_id", ["eq" => $rmaId]);
            foreach ($attachmentCollection->getItems() as $attachment):
                $this->attachmentRepository->deleteById($attachment["attachment_id"]);
            endforeach;
        } catch (\Exception $e) {
            throw new \Exception("Cannot delete attachment - " . $e->getMessage());
        }
    }
}
