<?php
/**
 * Author: Simone Monterubbiano <s.monterubbiano@drop.it>
 * Date: 04/03/2019
 * File name: RmaUpdated.php
 * Project: drop/module-rma-mirasvit
 */

namespace Drop\RmaMirasvit\Observer;

class RmaUpdated implements \Magento\Framework\Event\ObserverInterface
{
    /**
     * @var string
     */
    const QUEUE_NAME = "create_gsped_rma_shipment";

    /**
     * @var int
     */
    const STATUS_CLOSED = 5;

    /**
     * @var int
     */
    const STATUS_APPROVED = 2;

    /**
     * @var int
     */
    const STATUS_PENDING = 1;

    /**
     * @var \Rcason\Mq\Api\PublisherInterface
     */
    private $publisher;

    /**
     * @var \Drop\RmaMirasvit\Helper\Data
     */
    private $helper;

    /**
     * @var \Drop\RmaMirasvit\Model\QueueFactory
     */
    private $queueFactory;

    /**
     * @var \Magento\Sales\Api\OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var \Mirasvit\Rma\Model\RmaFactory
     */
    private $rmaFactory;

    /**
     * @var \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface
     */
    private $rmaManagement;

    /**
     * @var \Mirasvit\Rma\Helper\Mail
     */
    private $rmaMail;

    /**
     * @var \Mirasvit\Rma\Api\Repository\StatusRepositoryInterface
     */
    private $statusRepository;

    /**
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    private $timezone;

    /**
     * RmaUpdated constructor.
     * @param \Rcason\Mq\Api\PublisherInterface $publisher
     * @param \Drop\RmaMirasvit\Helper\Data $helper
     * @param \Drop\RmaMirasvit\Model\QueueFactory $queueFactory
     * @param \Magento\Sales\Api\Data\OrderInterface $orderRepository
     * @param \Mirasvit\Rma\Model\RmaFactory $rmaFactory
     * @param \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement
     */
    public function __construct(
        \Rcason\Mq\Api\PublisherInterface $publisher,
        \Drop\RmaMirasvit\Helper\Data $helper,
        \Drop\RmaMirasvit\Model\QueueFactory $queueFactory,
        \Magento\Sales\Api\OrderRepositoryInterface $orderRepository,
        \Mirasvit\Rma\Model\RmaFactory $rmaFactory,
        \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement,
        \Mirasvit\Rma\Helper\Mail $rmaMail,
        \Mirasvit\Rma\Api\Repository\StatusRepositoryInterface $statusRepository,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $timezone
    ) {
        $this->publisher = $publisher;
        $this->helper = $helper;
        $this->queueFactory = $queueFactory;
        $this->orderRepository = $orderRepository;
        $this->rmaFactory = $rmaFactory;
        $this->rmaManagement = $rmaManagement;
        $this->rmaMail = $rmaMail;
        $this->statusRepository = $statusRepository;
        $this->timezone = $timezone;
    }

    /**
     * If RMA status is closed delete pdf label from server
     * If job "create_gsped_rma_shipment" not exists create it
     * @param \Magento\Framework\Event\Observer $observer
     * @throws \Exception
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        $rma = $observer->getEvent()->getRma();

        $this->updateTimelineStatus(
            $rma
        );

        // If RMA status is closed delete labels from server
        if ($rma["status_id"]==self::STATUS_CLOSED) {
            $this->helper->deleteDocumentFiles($rma);
        }

        try {
            $order = $this->rmaManagement->getOrder($rma);
            $orderIncrementId = $order->getIncrementId();
        } catch (\Exception $e) {
            throw new \Exception("Cannot load order from RMA - ".$e->getMessage());
        }

        //set orderId if not set
        try {
            if ($rma->getOrderId() == 0 || is_null($rma->getOrderId())) {
                $rma->setOrderId($order->getId())->save();
            }
        } catch (\Exception $e) {
            throw new \Exception("Cannot set orderId for RMA - ".$e->getMessage());
        }

        if (
          !$this->helper->getIsNeedGenerateJobGspedShipment()
          ||
          ($this->helper->getIsNeedGenerateJobGspedShipment() && $this->helper->getIsNeedGenerateJobGspedOnlyIt() && $order->getShippingAddress()->getCountryId() != "IT")
        ) {
            $this->rmaSetApproved(
                $rma
            );

            $rmaUpdated = $this->preserveCustomerIdRma(
                $order,
                $rma
            );

            # force send email RMA ADMIN
            $status = $this->rmaManagement->getStatus($rmaUpdated);
            $adminMessage = $this->statusRepository->getAdminMessageForStore($status, $rmaUpdated->getStoreId());
            $this->rmaMail->sendNotificationAdminEmail($rmaUpdated, $adminMessage, true);

            return $rmaUpdated;
        }

        try {
            // There is no way to check if it's a new RMA, so check if job with message rma_id exists
            $collection = $this->queueFactory->create()->getCollection();
            $collection->addFieldToFilter("queue_name", ["eq" => "create_gsped_rma_shipment"]);
            $collection->addFieldToFilter("message_content", ["like" => "%" . $rma["rma_id"] . "%"]);
            if ($collection->count()) {
                return $this->preserveCustomerIdRma(
                    $order,
                    $rma
                );
            }
        } catch (\Exception $e) {
            throw new \Exception("Error during checking if job exists - ".$e->getMessage());
        }

        $this->publisher->publish(
            self::QUEUE_NAME,
            "Order #{$order->getIncrementId()} (RMA #{$rma->getIncrementId()}) - Create gsped shipment and save label to RMA",
            $rma["rma_id"]
        );
    }

    private function rmaSetApproved($rma)
    {
        try {
            if ($rma->getStatusId() == self::STATUS_PENDING) {
                $rma->setStatusId(self::STATUS_APPROVED)->save();
            }
        } catch (\Exception $e) {
            throw new \Exception("Cannot set status on RMA - ".$e->getMessage());
        }
    }

    private function updateTimelineStatus($rma)
    {
        try {
            $_na = $this->statusRepository->get($rma['status_id'])->getName() ?? null;

            if (empty($_na)) {
                return null;
            }

            $_ti = $this->timezone->date()->format('d/m/Y H:i:s');
            $_s = $rma->getDropRmaTimelineChangeStatus() . $_na . " > " . $_ti . PHP_EOL;
            $rma->setDropRmaTimelineChangeStatus($_s);
        } catch (\Exception $e) {
            return null;
        }
    }

    private function preserveCustomerIdRma($order, $rma)
    {
        if (!is_null($order->getCustomerId())) {
            $rma->setCustomerId(
                $order->getCustomerId()
            );
            $rma->save();
        }
        return $rma;
    }
}
