<?php
/**
 * Created by PhpStorm.
 * User: alberto
 * Date: 19/05/20
 * Time: 11.22
 */

namespace Drop\Pvs\Model\Consumer\Order;


class CheckCanceledConsumer implements \Rcason\Mq\Api\ConsumerInterface
{
    const DEFAULT_PVS_DIR = '/OUT/';
    const DEFAULT_PVS_PROCESSED_DIR = 'processed/';
    const NOT_SALABLE_STATUS = 'not_salable';
    const DEFAULT_PVS_CANCELED_ORDER_EMAIL_TEMPLATE_ID = 'pvs_canceled_order';
    const DEFAULT_PVS_RMA_REFUND_EMAIL_TEMPLATE_ID = 'pvs_rma_refund';
    const PROCESSED_ORDER_STATUS = [
        'not_salable',
        'closed',
    ];

    private $helper;
    private $ftp;
    private $cancelOrRefund;
    private $order;
    private $creditmemoFactory;
    private $productRepository;
    private $stockRegistry;

    private $storeId;
    private $emailEnabled;

    /**
     * CheckCanceledConsumer constructor.
     * @param \Drop\Pvs\Helper\Data $helper
     * @param \Drop\Pvs\Helper\Ftp $ftp
     * @param \Drop\Pvs\Model\Adyen\CancelOrRefund $cancelOrRefund
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @param \Magento\Sales\Model\Order\CreditmemoFactory $creditmemoFactory
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\CatalogInventory\Api\StockRegistryInterface $stockRegistry
     */
    public function __construct(
        \Drop\Pvs\Helper\Data $helper,
        \Drop\Pvs\Helper\Ftp $ftp,
        \Drop\Pvs\Model\Adyen\CancelOrRefund $cancelOrRefund,
        \Magento\Sales\Api\Data\OrderInterface $order,
        \Magento\Sales\Model\Order\CreditmemoFactory $creditmemoFactory,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\CatalogInventory\Api\StockRegistryInterface $stockRegistry
    )
    {
        $this->helper = $helper;
        $this->ftp = $ftp;
        $this->cancelOrRefund = $cancelOrRefund;
        $this->order = $order;
        $this->creditmemoFactory = $creditmemoFactory;
        $this->productRepository = $productRepository;
        $this->stockRegistry = $stockRegistry;

        $this->emailEnabled = $this->helper->checkIfPvsEmailsIsEnabled();
    }

    /**
     * @param $filename
     * @throws \Exception
     */
    public function process($filename)
    {
        $filePaths = $this->ftp->getFilesList(self::DEFAULT_PVS_DIR, $filename);
        foreach ($filePaths as $filePath) {
            $fileContent = $this->ftp->readFileFromFtp($filePath);
            $ordersList = $this->helper->getFormattedArrayByXordcStockFileContent($fileContent);
            foreach ($ordersList as $orderIncrementId => $pvsOrder) {
                $order = $this->order->loadByIncrementId($orderIncrementId);
                if (!$this->canProcess($order)) {
                    continue;
                }

                switch ($this->canCancelOrder($pvsOrder, $order)):
                    case false:
                        $this->cancelPartialOrder($pvsOrder, $order);
                        break;
                    case true:
                        $this->cancelTotalOrder($order);
                        break;
                endswitch;

                if ($this->helper->canDecreaseQty()) {
                    $this->decreaseProductQty($pvsOrder);
                }
            }

            // move to processed
            $filePathExplode = explode('/', $filePath);
            $this->ftp->moveFtpFile($filePath, self::DEFAULT_PVS_DIR . self::DEFAULT_PVS_PROCESSED_DIR . end($filePathExplode));
        }
    }

    /**
     * @param $pvsOrder
     * @param $order
     * @return bool
     */
    public function canCancelOrder($pvsOrder, $order)
    {
        foreach ($pvsOrder['items'] as $pvsItem) {
            if ($pvsItem['qta_buoni'] > 0) {
                // in questo caso significa che almeno 1 quantità di 1 articolo è vendibile => non è una cancellazione totale
                return false;
            }
        }

        foreach ($order->getAllVisibleItems() as $item) {
            if (!in_array($item->getSku(), $pvsOrder['skus'])) {
                // in questo caso significa che almeno 1 articolo dell'ordine non ha quantità non vendibili => non è una cancellazione totale
                return false;
            }
        }

        // se arrivo alla fine signifca che gli articoli nel file sono tutti quelli dell'ordine, e tutti gli articoli sono invendibili per la quantità totale
        return true;
    }

    /**
     * @param $order
     * @throws \Exception
     */
    public function cancelTotalOrder($order)
    {
        $comment = "The order is not salable because all products are out of stock/damaged";
        $order->addStatusToHistory(self::NOT_SALABLE_STATUS, $comment);
        try {
            $order->save();
        } catch (\Exception $e) {
            throw new \Exception("[PVS][UPDATE_ORDER] #{$order->getIncrementId()}");
        }

        // crezione nota di credito
        $creditMemo = $this->creditmemoFactory->createByOrder($order, $order->getData());
        $creditMemo->setOrder($order);
        $creditMemo->setState(\Magento\Sales\Model\Order\Creditmemo::STATE_OPEN);
        try {
            $creditMemo->save();
        } catch (\Exception $e) {
            throw new \Exception("[PVS][CREATE_CMEMO] Error to create creditmemo for order #{$order->getIncrementId()}");
        }

        // se non è stato pagato con adyen invio la mail al cc per procedere manualmente al rimborso
        if (!strpos($order->getPayment()->getMethod(), 'adyen')) {
            // invio email
            $templateVars = [
                'title'          => "Order #{$order->getIncrementId()} is not salable, the refund for creditmemo #{$creditMemo->getIncrementId()} can be processed",
                'order'          => $order->getIncrementId(),
                'order_date'     => $order->getCreatedAt(),
                'payment_method' => $order->getPayment()->getMethod(),
                'grand_total'    => $creditMemo->getGrandTotal(),
                'note'           => $comment,
            ];
            if (!$this->emailEnabled) {
                $this->helper->sendMail(self::DEFAULT_PVS_RMA_REFUND_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getRmaEmails($order->getStoreId()));
            }
            return;
        }

        // rimborso (uso il cancelOrRefund perchè il refund va in errore se non viene trovato il pagamento)
        $this->cancelOrRefund->process($order);

        // aggiornamento stato nota di credito
        $creditMemo->setState(\Magento\Sales\Model\Order\Creditmemo::STATE_REFUNDED);
        try {
            $creditMemo->save();
        } catch (\Exception $e) {
            throw new \Exception("[PVS][CREATE_CMEMO] Error to update creditmemo state for order #{$order->getIncrementId()}");
        }
    }

    /**
     * @param $pvsOrder
     * @param $order
     * @throws \Exception
     */
    public function cancelPartialOrder($pvsOrder, $order)
    {
        $comment = "The order is not salable only for these products:";
        foreach ($pvsOrder['items'] as $pvsItem) {
            if ($pvsItem['qta_ann2'] > 0) {
                $comment .= "<br/>product " . $pvsItem["cod_art"] . " available only for " . $pvsItem["qta_buoni"] . " qty (" . $pvsItem["qta_riga"] . " required)";
            }
        }

        $order->setStatus(self::NOT_SALABLE_STATUS);
        $order->addStatusToHistory(self::NOT_SALABLE_STATUS, $comment);
        try {
            $order->save();
        } catch (\Exception $e) {
            throw new \Exception("[PVS][UPDATE_ORDER] #{$order->getIncrementId()}");
        }

        // invio email
        $templateVars = [
            'title'   => "Order #{$order->getIncrementId()} is not all salable",
            'message' => $comment
        ];
        if (!$this->emailEnabled) {
            $this->helper->sendMail(self::DEFAULT_PVS_CANCELED_ORDER_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getCancelOrderEmails($order->getStoreId()));
        }
    }

    /**
     * @param $pvsOrder
     * @throws \Exception
     */
    public function decreaseProductQty($pvsOrder)
    {
        foreach ($pvsOrder['items'] as $item) {
            // se l'articolo non è difettato, salto
            if ((float)$item['qta_ann2'] == 0) {
                continue;
            }

            if (!$product = $this->productRepository->get($item['cod_art'])) {
                throw new \Exception("[PVS][DECREASE_PRODUCT_QTY] product {$item['cod_art']} not found");
            }

            try {
                $stockItem = $this->stockRegistry->getStockItemBySku($product->getSku());
                $qty = (float)$stockItem->getQty() - (float)$item['qta_ann2'];
                $isInStock = 1;
                if ($qty < 0) {
                    $qty = 0;
                    $isInStock = 0;
                }
                $stockItem->setQty($qty);
                $stockItem->setIsInStock($isInStock);
                $this->stockRegistry->updateStockItemBySku($product->getSku(), $stockItem);
            } catch (\Exception $e) {
                throw new \Exception("[PVS][DECREASE_PRODUCT_QTY] Error while updating stock product {$item['cod_art']} - {$e->getMessage()}");
            }
        }
    }

    /**
     * @param $order
     * @return bool
     */
    public function canProcess($order)
    {
        // se l'ordine è in stato non vendibile o chiuso salto la riga
        if (in_array($order->getStatus(), self::PROCESSED_ORDER_STATUS)) {
            return false;
        }

        return true;
    }
}