<?php
/**
 * Created by PhpStorm.
 * User: alberto
 * Date: 25/08/20
 * Time: 10.52
 */

namespace Drop\Pvs\Model\Consumer\Report;

use Google_Client;
use Google_Service_Analytics;

class Daily implements \Rcason\Mq\Api\ConsumerInterface
{
    const INITIAL_CURRENCY = 'EUR';
    const DEFAULT_PVS_DAILY_REPORT_EMAIL_TEMPLATE_ID = 'pvs_daily_report';

    protected $helper;
    protected $orderCollectionFactory;
    protected $country;
    protected $helperBackend;
    private $rates;

    public function __construct(
        \Drop\Pvs\Helper\Data $helper,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory,
        \Magento\Directory\Model\Country $country,
        \Magento\Backend\Helper\Data $helperBackend
    )
    {
        $this->helper = $helper;
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->country = $country;
        $this->helperBackend = $helperBackend;
    }

    /**
     * @param $date
     * @throws \Exception
     */
    public function process($date)
    {
        $templateVars = $this->getData($date);

        if ($templateVars['orders'] == 0) {
            $templateVars['note'] = 'There are no orders for this date';
        }

        $this->helper->sendMail(self::DEFAULT_PVS_DAILY_REPORT_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getReportEmails());
    }

    /**
     * @param $date
     * @param bool $getHtml
     * @return array
     * @throws \Exception
     */
    public function getData($date, $getHtml = true)
    {
        $ordersWithoutShipCost = 0;
        $shippingCostAmount = 0;
        $shippingNetCostAmount = 0;
        $piecesSold = 0;
        $paidAmount = 0;
        $netPaidAmount = 0;
        $orderDetails = [];
        $bestSellers = [];
        $customersEmails = [];
        $customers = 0;

        $this->setRates($date);
        $orders = $this->getOrderCollection($date);
        foreach ($orders as $order) {
            if (!in_array($order->getCustomerEmail(), $customersEmails)) {
                $customersEmails[] = $order->getCustomerEmail();
                $customers++;
            }
            // ----- ORDER TOTALS -----
            $totals = $this->getOrderCalculatedTotals($order);
            $shippingCostAmount += $totals['shippingAmount'];
            $shippingNetCostAmount += $totals['shippingNetAmount'];
            $paidAmount += $totals['grandTotal'];
            $netPaidAmount += $totals['subTotal'];
            if ($totals['shippingAmount'] == 0) {
                $ordersWithoutShipCost++;
            }

            // ----- PRODUCT DETAILS -----
            foreach ($order->getAllVisibleItems() as $item) {
                $itemTotals = $this->getItemCalculatedTotals($order, $item);
                $formatTaxPercent = number_format($item->getTaxPercent(), 2);
                $sku = (!$getHtml) ? $item->getSku() : '<a href="' . $this->helperBackend->getUrl('sales/order/view', ['order_id' => $order->getId()]) . '">' . $item->getSku() . '</a>';
                $orderDetails[] = "{$sku} - € {$itemTotals['grandTotal']} Gross (€ {$itemTotals['subTotal']} Net, {$formatTaxPercent}% VAT) - {$this->getCountryName($order->getBillingAddress()->getCountryId())}";
//                $bestSellerKey = $item->getSku() . ' - ' . $item->getName() . ' - ' . $item->getProduct()->getResource()->getAttribute('color')->getFrontend()->getValue($item->getProduct());
                if ($item->getProduct()) {
                    $bestSellerKey = $item->getProduct()->getBaseSku() . ' - ' . $item->getName();
                } else {
                    continue;
                }
                $bestSellers[$bestSellerKey] = isset($bestSellers[$bestSellerKey]) ? $bestSellers[$bestSellerKey] + (float)$item->getQtyOrdered() : (float)$item->getQtyOrdered();
                $piecesSold++;
            }
        }

        if (count($orders) == 0) {
            return ['orders' => 0];
        }

        $analyticsData = $this->getAnalyticsData();
        $cr = $this->getConversionRate($orders, $analyticsData['sessions']);

        return [
            'report_date'              => $date,
            'orders'                   => count($orders),
            'orders_without_ship_cost' => $ordersWithoutShipCost,
            'shipping_cost_amount'     => "€ " . number_format($shippingCostAmount, 2, ',', '.'),
            'shipping_net_cost_amount' => "€ " . number_format($shippingNetCostAmount, 2, ',', '.'),
            'pieces_sold'              => $piecesSold,
            'paid_amount'              => "€ " . number_format($paidAmount, 2, ',', '.'),
            'net_paid_amount'          => "€ " . number_format($netPaidAmount, 2, ',', '.'),
            'average_order'            => "€ " . number_format($paidAmount / count($orders), 2, ',', '.'),
            'best_sellers'             => ($getHtml) ? $this->getHtmlListFormat($this->formatBestSellers($bestSellers)) : $bestSellers,
            'order_details'            => ($getHtml) ? $this->getHtmlListFormat($orderDetails) : $orderDetails,
            'sessions'                 => $this->helper->checkIfAnalyticsIsEnabled() ? $analyticsData['sessions'] : '',
            'conversion_rate'          => $this->helper->checkIfAnalyticsIsEnabled() ? $cr : '',
            'unique_users'             => $analyticsData['users'],
        ];
    }

    /**
     * @param $order
     * @return array
     */
    public function getOrderCalculatedTotals($order)
    {
        $shippingAmount = (float)$order->getBaseShippingAmount();
        $shippingTaxAmount = (float)$order->getBaseShippingTaxAmount();
        $grandTotal = (float)$order->getBaseGrandTotal();
        $taxAmount = (float)$order->getBaseTaxAmount();
        if ($this->rates && $order->getBaseCurrencyCode() != self::INITIAL_CURRENCY) {
            $shippingAmount = $shippingAmount * $this->rates[$order->getBaseCurrencyCode()];
            $shippingTaxAmount = $shippingTaxAmount * $this->rates[$order->getBaseCurrencyCode()];
            $grandTotal = $grandTotal * $this->rates[$order->getBaseCurrencyCode()];
            $taxAmount = $taxAmount * $this->rates[$order->getBaseCurrencyCode()];
        }

        return [
            'shippingAmount'    => $shippingAmount,
            'shippingTaxAmount' => $shippingTaxAmount,
            'shippingNetAmount' => $shippingAmount - $shippingTaxAmount,
            'grandTotal'        => $grandTotal,
            'taxAmount'         => $taxAmount,
            'subTotal'          => $grandTotal - $taxAmount,
        ];
    }

    /**
     * @param $order
     * @param $item
     * @return array
     */
    public function getItemCalculatedTotals($order, $item)
    {
        $grandTotal = $item->getBaseRowTotalInclTax();
        $taxAmount = $item->getBaseTaxAmount();

        if ($this->rates && $order->getBaseCurrencyCode() != self::INITIAL_CURRENCY) {
            $grandTotal = $grandTotal * $this->rates[$order->getBaseCurrencyCode()];
            $taxAmount = $taxAmount * $this->rates[$order->getBaseCurrencyCode()];
        }

        return [
            'grandTotal' => number_format($grandTotal, 2, ',', '.'),
            'taxAmount'  => number_format($taxAmount, 2, ',', '.'),
            'subTotal'   => number_format($grandTotal - $taxAmount, 2, ',', '.'),
        ];
    }

    /**
     * @param $date
     * @return \Magento\Sales\Model\ResourceModel\Order\Collection
     */
    public function getOrderCollection($date)
    {
        $collection = $this->orderCollectionFactory->create()
            ->addAttributeToSelect('*')
            ->addFieldToFilter('created_at', ['gteq' => $date . ' 00:00:00'])
            ->addFieldToFilter('created_at', ['lteq' => $date . ' 23:59:59'])
            ->addFieldToFilter('status', ['in' => $this->helper->getAcceptedReportOrderStatuses()]);

        return $collection;

    }

    /**
     * @param $countryCode
     * @return string
     * @throws \Exception
     */
    public function getCountryName($countryCode)
    {
        try {
            $country = $this->country->loadByCode($countryCode);
        } catch (\Exception $e) {
            throw new \Exception("[PVS][DAILY_REPORT][COUNTRY] errore recupero info country_id {$countryCode} - error: {$e->getMessage()}");
        }

        return $country->getName();
    }

    /**
     * @param $date
     * @throws \Exception
     */
    public function setRates($date)
    {
        $filePath = "https://openexchangerates.org/api/historical/" . $date . ".json?app_id=" . $this->helper->getRatesAppId();
        $response = file_get_contents($filePath);
        $response = json_decode($response);

        $initial = self::INITIAL_CURRENCY;
        $eurRate = $response->rates->$initial;

        foreach ($response->rates as $key => $value) {
            if ($key == $initial) {
                continue;
            }

            $convertedRate = 1 / $value * $eurRate;
            $this->rates[$key] = $convertedRate;
        }

        if (!count($this->rates)) {
            throw new \Exception("errore download tassi di cambio");
        }
    }

    /**
     * @param $bestSellers
     * @return array
     */
    public function formatBestSellers($bestSellers)
    {
        arsort($bestSellers);
        $bestSellers = array_slice($bestSellers, 0, $this->helper->getBestSellersLimit());
        $formattedBestSeller = [];
        foreach ($bestSellers as $sku => $qty) {
            $formattedBestSeller[] = $sku . " - " . $qty;
        }

        return $formattedBestSeller;
    }

    /**
     * @param $array
     * @return string
     */
    public function getHtmlListFormat($array)
    {
        return "<li>" . implode('</li><li>', $array) . "</li>";
    }

    /**
     * @return array|string
     */
    public function getAnalyticsData()
    {
        if (!$this->helper->checkIfAnalyticsIsEnabled()) {
            return '';
        }

        $d = json_decode($this->helper->getAnalyticsPrivateKey());
        $key = [
            "type"           => "service_account",
            "project_id"     => $this->helper->getAnalyticsProjectId(),
            "private_key_id" => $this->helper->getAnalyticsPrivateKeyId(),
            "private_key"    => $d->private_key,
            "client_email"   => $this->helper->getAnalyticsClientEmail(),
            "client_id"      => $this->helper->getAnalyticsClientId(),
        ];

        try {
            $client = new Google_Client();
            $client->setScopes([
                Google_Service_Analytics::ANALYTICS_READONLY,
            ]);
            $client->setAuthConfig($key);
            $googleService = new Google_Service_Analytics($client);
            $data = $googleService->data_ga->get(
                'ga:' . $this->helper->getAnalyticsPageId(),
                'yesterday',
                'yesterday',
                'ga:sessions,ga:users');
        } catch (\Exception $e) {
            echo $e->getMessage();
            return '';
        }

        if (!$data ||
            !isset($data->totalsForAllResults['ga:sessions']) ||
            !isset($data->totalsForAllResults['ga:users'])
        ) {
            return '';
        }

        return [
            'sessions' => $data->totalsForAllResults['ga:sessions'],
            'users'    => $data->totalsForAllResults['ga:users'],
        ];
    }

    /**
     * @param $orders
     * @param $sessions
     * @return string
     */
    public function getConversionRate($orders, $sessions)
    {
        if (!$this->helper->checkIfAnalyticsIsEnabled()) {
            return '';
        }

        if (!$sessions || $sessions == 0) {
            return '';
        }

        return round((count($orders) / $sessions) * 100, 2) . " %";
    }
}
