<?php
/**
 * Created by PhpStorm.
 * User: alberto
 * Date: 05/06/20
 * Time: 16.20
 */

namespace Drop\Pvs\Model\Consumer\Order\Rma;


class ManageReturnedConsumer
{
    const XML_GENERAL_COUNTRY_EU_COUNTRIES = 'general/country/eu_countries';
    const DEFAULT_PVS_DIR = '/OUT/';
    const DEFAULT_PVS_PROCESSED_DIR = 'processed/';
    const RMA_RETURN_ERROR = 17;
    const RMA_REFUNDED = 'refunded';
    const RMA_ALLOW_REFUND = 'allow_refund';
    const RMA_RESOLUTION_EXCHANGE_ID = 1;
    const DEFAULT_PVS_WRONG_RMA_EMAIL_TEMPLATE_ID = 'pvs_wrong_rma';
    const DEFAULT_PVS_RMA_REFUND_EMAIL_TEMPLATE_ID = 'pvs_rma_refund';
    const DEFAULT_PVS_RMA_EXCHANGE_EMAIL_TEMPLATE_ID = 'pvs_rma_exchange';
    const DEFAULT_PVS_REFUND_INFO_EMAIL_TEMPLATE_ID = 'pvs_info_success_refund';
    const ACCEPTED_RMA_COD_MCON = ['001', '020'];

    private $order;
    private $creditmemoSender;
    private $creditmemoLoader;
    private $creditmemoManagement;
    private $rmaManagement;
    private $statusRepository;
    private $itemListBuilder;
    private $helper;
    private $ftp;
    private $refund;
    private $euCountries;

    private $storeId;
    private $emailEnabled;

    /**
     * ManageReturnedConsumer constructor.
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @param \Magento\Sales\Model\Order\Email\Sender\CreditmemoSender $creditmemoSender
     * @param \Magento\Sales\Controller\Adminhtml\Order\CreditmemoLoader $creditmemoLoader
     * @param \Magento\Sales\Api\CreditmemoManagementInterface $creditmemoManagement
     * @param \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement
     * @param \Mirasvit\Rma\Api\Repository\StatusRepositoryInterface $statusRepository
     * @param \Mirasvit\Rma\Service\Item\ItemListBuilder $itemListBuilder
     * @param \Drop\Pvs\Helper\Data $helper
     * @param \Drop\Pvs\Helper\Ftp $ftp
     * @param \Drop\Pvs\Model\Adyen\Refund $refund
     */
    public function __construct(
        \Magento\Sales\Api\Data\OrderInterface $order,
        \Magento\Sales\Model\Order\Email\Sender\CreditmemoSender $creditmemoSender,
        \Magento\Sales\Controller\Adminhtml\Order\CreditmemoLoader $creditmemoLoader,
        \Magento\Sales\Api\CreditmemoManagementInterface $creditmemoManagement,
        \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement,
        \Mirasvit\Rma\Api\Repository\StatusRepositoryInterface $statusRepository,
        \Mirasvit\Rma\Service\Item\ItemListBuilder $itemListBuilder,
        \Drop\Pvs\Helper\Data $helper,
        \Drop\Pvs\Helper\Ftp $ftp,
        \Drop\Pvs\Model\Adyen\Refund $refund
    )
    {
        $this->order = $order;
        $this->creditmemoSender = $creditmemoSender;
        $this->creditmemoLoader = $creditmemoLoader;
        $this->creditmemoManagement = $creditmemoManagement;
        $this->rmaManagement = $rmaManagement;
        $this->statusRepository = $statusRepository;
        $this->itemListBuilder = $itemListBuilder;
        $this->helper = $helper;
        $this->ftp = $ftp;
        $this->refund = $refund;

        $this->emailEnabled = $this->helper->checkIfPvsEmailsIsEnabled();
        $this->euCountries = explode(',', $this->helper->getConfigValue(self::XML_GENERAL_COUNTRY_EU_COUNTRIES));
    }

    /**
     * @param $filename
     * @throws \Exception
     */
    public function process($filename)
    {
        $filePaths = $this->ftp->getFilesList(self::DEFAULT_PVS_DIR, $filename);
        foreach ($filePaths as $filePath) {
            $fileContent = $this->ftp->readFileFromFtp($filePath);
            $processedRmas = $this->helper->getFormattedArrayByXdoceTesFileContentRmaFilter($fileContent);
            if (count($processedRmas) == 0) {
                continue;
            }

            foreach ($processedRmas as $orderIncrementId => $items) {
                $this->manageOrder($orderIncrementId, $items);
            }

            // move to processed
            $filePathExplode = explode('/', $filePath);
            $this->ftp->moveFtpFile($filePath, self::DEFAULT_PVS_DIR . self::DEFAULT_PVS_PROCESSED_DIR . end($filePathExplode));
        }
    }

    /**
     * @param $orderIncrementId
     * @param $returnedItems
     * @throws \Exception
     */
    public function manageOrder($orderIncrementId, $returnedItems)
    {
        // controllo se esiste l'ordine
        if (!$order = $this->order->loadByIncrementId($orderIncrementId)) {
            throw new \Exception("[PVS][MANAGE_RETURNED_RMA] Order #{$orderIncrementId} not found");
        }

        // fare un check se può essere fatto o meno il rimborso (ad esempio controllo se c'è già stato un rimborso)
        if (!$this->canRefund($order, $returnedItems)) {
            return;
        }

        $checkedProblems = false;
        $messageHtml = '';
        // controllo se il reso ricevuto ha una richiesta di reso => gestione manuale
        if (!$rmas = $this->rmaManagement->getRmasByOrder($order)) {
            $checkedProblems = true;
            $messageHtml = "<p>Received rma has no request</p>";
        }

        // controllo se l'articolo rientrato è quello inserito nella richiesta di reso ricevuta e se fa parte dell'ordine => gestione manuale
        $rma = $this->getRmaFromCollection($rmas, $returnedItems);
        if ($rmas && !$rma) {
            // caso in cui ci sono richieste di reso, ma nessuna di queste coincide con il numero passato da pvs
            throw new \Exception("[PVS][MANAGE_RETURNED_RMA] Rma for order #{$orderIncrementId} not found");
        }

        if ($rma && !$this->checkIfReturnedItemsIsInRequest($rma, $returnedItems)) {
            // cambio lo stato del reso
            $status = $this->statusRepository->getByCode(self::RMA_RETURN_ERROR);
            $rma->setStatusId($status->getId());
            $rma->save();

            $checkedProblems = true;
            $messageHtml = "<p>Received item is not the same as the request</p>";
        }


        // controllo se l'articolo che torna è vendibile, se no => gestione manuale
        if (!$this->checkIfReturnedItemsIsSalable($returnedItems)) {
            $checkedProblems = true;
            $messageHtml = "<p>Received item is not salable</p>";
        }

        if ($checkedProblems) {
            $templateVars = [
                'title'   => "Error in rma for order #{$order->getIncrementId()}",
                'message' => $messageHtml
            ];
            if (!$this->emailEnabled) {
                $this->helper->sendMail(self::DEFAULT_PVS_WRONG_RMA_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getRmaEmails($order->getStoreId()));
            }

            return;
        }

        // nel caso sia tutto ok creo la nota di credito
        $creditmemo = $this->createCreditmemo($order, $returnedItems);

        // controllo se è stato richiesto un cambio => gestione manuale
        if ($this->checkIfRequestChange($rma, $returnedItems)) {
            // invio email
            $templateVars = [
                'title'      => "The refund contain an exchange request",
                'order'      => $order->getIncrementId(),
                'order_date' => $order->getCreatedAt(),
                'rma'        => $rma->getIncrementId(),
            ];
            if (!$this->emailEnabled) {
                $this->helper->sendMail(self::DEFAULT_PVS_RMA_EXCHANGE_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getRmaEmails($order->getStoreId()));
            }
            return;
        }

        // se non è stato pagato con adyen o se è EXTRAUE (discorso dazi) invio la mail al cc per procedere manualmente al rimborso
        if (
            !strpos($order->getPayment()->getMethod(), 'adyen')
            || !in_array($order->getBillingAddress()->getCountryId(), $this->euCountries)
        ) {
            $creditmemo->setState(\Magento\Sales\Model\Order\Creditmemo::STATE_OPEN);
            try {
                $creditmemo->save();
            } catch (\Exception $e) {
                throw new \Exception("[PVS][MANAGE_RETURNED_RMA] Error on creditmemo #{$creditmemo->getIncrementId()} save - {$e->getMessage()}");
            }
            $status = $this->statusRepository->getByCode(self::RMA_ALLOW_REFUND);
            $rma->setStatusId($status->getId());
            $rma->save();
            // invio email
            $templateVars = [
                'title'          => "The refund for creditmemo #{$creditmemo->getIncrementId()} can be processed",
                'order'          => $order->getIncrementId(),
                'order_date'     => $order->getCreatedAt(),
                'payment_method' => $order->getPayment()->getMethod(),
                'grand_total'    => $creditmemo->getGrandTotal(),
                'note'           => (!in_array($order->getBillingAddress()->getCountryId(), $this->euCountries)) ? 'N.B. check duty import' : '',
            ];
            if (!$this->emailEnabled) {
                $this->helper->sendMail(self::DEFAULT_PVS_RMA_REFUND_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getRmaEmails($order->getStoreId()));
            }
            return;
        }

        // rimborso
        $this->refund->process($order, $creditmemo->getGrandTotal());

        // cambio lo stato del reso in rimborsato
        $status = $this->statusRepository->getByCode(self::RMA_REFUNDED);
        $rma->setStatusId($status->getId());
        $rma->save();

        // invio mail al cc per info
        $templateVars = [
            'title'          => "Creditmemo #{$creditmemo->getIncrementId()} has been refunded",
            'order'          => $order->getIncrementId(),
            'order_date'     => $order->getCreatedAt(),
            'payment_method' => $order->getPayment()->getMethod(),
            'grand_total'    => $creditmemo->getGrandTotal(),
            'note'           => '',
        ];
        if (!$this->emailEnabled) {
            $this->helper->sendMail(self::DEFAULT_PVS_REFUND_INFO_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getRmaEmails($order->getStoreId()));
        }
    }

    /**
     * @param $rmas
     * @param $returnedItems
     * @return null
     */
    public function getRmaFromCollection($rmas, $returnedItems)
    {
        $rma = null;
        $firstReturnedItem = reset($returnedItems['items']);
        foreach ($rmas as $_rma) {
            if ($_rma->getIncrementId() != $firstReturnedItem['bolla']) {
                continue;
            }
            $rma = $_rma;
        }

        return $rma;
    }

    /**
     * @param $order
     * @param $returnedItems
     * @return bool
     */
    public function canRefund($order, $returnedItems)
    {
        // controllare se ci sono articoli rimborsati
        foreach ($order->getAllVisibleItems() as $item) {
            if (
                in_array($item->getSku(), $returnedItems['skus']) &&
                $returnedItems['items'][$item->getSku()]['qta_versata'] > ($item->getQtyInvoiced() - $item->getQtyRefunded())
            ) {
                // l'articolo reso è già stato rimborsato o la quantità resa è superiore a quella fatturata
                return false;
            }
        }

        return true;
    }

    /**
     * @param $order
     * @param $returnedItems
     * @return mixed
     * @throws \Exception
     */
    public function createCreditmemo($order, $returnedItems)
    {
        $creditMemoData = [
            'do_offline'          => 1,
            'shipping_amount'     => 0,
            'adjustment_positive' => 0,
            'adjustment_negative' => 0,
            'comment_text'        => '',
            'send_email'          => $this->helper->getSendCreditmemoEmailToCustomer($order->getStoreId()),
        ];

        $creditMemoItems = [];
        foreach ($order->getAllVisibleItems() as $item) {
            if (!in_array($item->getSku(), $returnedItems['skus'])) {
                continue;
            }

            $creditMemoItems[$item->getId()] = ['qty' => $returnedItems['items'][$item->getSku()]['qta_versata']];
        }
        $creditMemoData['items'] = $creditMemoItems;

        $this->creditmemoLoader->setOrderId($order->getId());
        $this->creditmemoLoader->setCreditmemo($creditMemoData);

        try {
            $creditmemo = $this->creditmemoLoader->load();
            if ($creditmemo) {
                if (!empty($creditMemoData['comment_text'])) {
                    $creditmemo->addComment(
                        $creditMemoData['comment_text'],
                        isset($creditMemoData['comment_customer_notify']),
                        isset($creditMemoData['is_visible_on_front'])
                    );

                    $creditmemo->setCustomerNote($creditMemoData['comment_text']);
                    $creditmemo->setCustomerNoteNotify(isset($creditMemoData['comment_customer_notify']));
                }


                $creditmemo->getOrder()->setCustomerNoteNotify(!empty($creditMemoData['send_email']));
                $this->creditmemoManagement->refund($creditmemo, (bool)$creditMemoData['do_offline']);

                if (!empty($creditMemoData['send_email'])) {
                    $this->creditmemoSender->send($creditmemo);
                }

                return $creditmemo;
            }
        } catch (\Exception $e) {
            throw new \Exception("[PVS][MANAGE_RETURNED_RMA] Can't create Creditmemo for #{$order->getIncrementId()} - {$e->getMessage()}");
        }
    }

    /**
     * @param $rma
     * @param $returnedItems
     * @return bool
     */
    public function checkIfReturnedItemsIsInRequest($rma, $returnedItems)
    {
        $rmaItems = $this->itemListBuilder->getRmaItems($rma);
        $itemCount = 0;
        foreach ($rmaItems as $rmaItem) {
            if (!in_array($rmaItem->getProductSku(), $returnedItems['skus'])) {
                continue;
            }

            $itemCount++;
        }

        if ($itemCount == count($returnedItems['skus'])) {
            // è nella richiesta di reso e quindi nell'ordine
            return true;
        }

        return false;
    }

    /**
     * @param $returnedItems
     * @return bool
     */
    public function checkIfReturnedItemsIsSalable($returnedItems)
    {
        foreach ($returnedItems['items'] as $item) {
            if (!in_array($item['cod_mcon'], self::ACCEPTED_RMA_COD_MCON)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param $rma
     * @param $returnedItems
     * @return bool
     */
    public function checkIfRequestChange($rma, $returnedItems)
    {

        $rmaItems = $this->itemListBuilder->getRmaItems($rma);
        foreach ($rmaItems as $rmaItem) {
            if (!in_array($rmaItem->getProductSku(), $returnedItems['skus'])) {
                continue;
            }

            if ($rmaItem->getResolutionId() == self::RMA_RESOLUTION_EXCHANGE_ID) {
                return true;
            }
        }

        return false;
    }
}