<?php
/**
 * Created by PhpStorm.
 * User: alberto
 * Date: 12/05/20
 * Time: 11.48
 */

namespace Drop\Pvs\Model\Consumer\Order\Invoice;


class PayConsumer implements \Rcason\Mq\Api\ConsumerInterface
{
    const DEFAULT_PVS_COURIERS_DIR = '/OUT/VETTORI/';
    const DEFAULT_PVS_COURIERS_DIR_NAME = 'OUT/VETTORI';
    const DEFAULT_PVS_COURIERS_OLD_DIR = 'OLD/';
    const ORDER_STATUS_CAN_PICK_UP_PVS = 'can_pick_up_PVS';
    const DEFAULT_PVS_RECEIVED_SPECIAL_ITEMS_EMAIL_TEMPLATE_ID = 'pvs_received_special_items_order';
    const PROFORMA = 'Proforma';
    const ACCOMPAGNATORIA = 'Accompagnatoria';

    private $invoice;
    private $state;
    private $backendUrlManager;
    private $helper;
    private $ftp;
    private $pdf;
    private $templateRepository;
    private $invoiceFactory;
    private $messageFactory;

    public function __construct(
        \Magento\Framework\App\State $state,
        \Magento\Sales\Model\Order\Invoice $invoice,
        \Magento\Backend\Model\Url $backendUrlManager,
        \Drop\Pvs\Helper\Data $helper,
        \Drop\Pvs\Helper\Ftp $ftp,
        \Drop\Pdfgenerator\Helper\Pdf $pdf,
        \Eadesigndev\Pdfgenerator\Model\PdfgeneratorRepository $templateRepository,
        \Magento\Sales\Model\ResourceModel\Order\Invoice\CollectionFactory $invoiceFactory,
        \Magento\GiftMessage\Model\MessageFactory $messageFactory
    )
    {
        $this->state = $state;
        $this->invoice = $invoice;
        $this->backendUrlManager = $backendUrlManager;
        $this->helper = $helper;
        $this->ftp = $ftp;
        $this->pdf = $pdf;
        $this->templateRepository = $templateRepository;
        $this->invoiceFactory = $invoiceFactory;
        $this->messageFactory = $messageFactory;
    }

    /**
     * @param $incrementId
     * @throws \Exception
     */
    public function process($incrementId)
    {
        try {
            // this tosses an error if the areacode is not set.
            $this->state->getAreaCode();
        } catch (\Exception $e) {
            $this->state->setAreaCode('adminhtml');
        }

        if (!$this->helper->checkIfPvsIsEnabled()) {
            return;
        }

        // If invoice increment_id is empty, return
        if (empty($incrementId)) {
            throw new \Exception("[PVS][SAVE_DOC_FILE_FTP] Increment Id is empty");
        }

        // Load invoice by incrementId
        try {
            $_invoice = $this->invoice->loadByIncrementId((string)$incrementId);

            $invoice = $this->invoiceFactory->create()
                ->addAttributeToSelect('*')
                ->addAttributeToFilter('entity_id', ['eq' => $_invoice->getEntityId()])
                ->getFirstItem();

        } catch (\Exception $e) {
            throw new \Exception("[PVS][SAVE_DOC_FILE_FTP] Cannot load invoice by incrementId #{$incrementId}: {$e->getMessage()}");
        }

        // If order is not valid, return
        if (!is_object($invoice)) {
            throw new \Exception("[PVS][SAVE_DOC_FILE_FTP] Invoice is not an object");
        }

        $order = $invoice->getOrder();
        if ($this->helper->checkIsGiftCard($order)) {
            // non devo esportare nulla in quanto giftcard
            return;
        }

        if ($this->helper->checkIfSpecialItemsIsEnabled() && $this->helper->checkIfOrderHasSpecialItems($order)) {
            $skus = [];
            foreach ($order->getAllVisibleItems() as $item) {
                if (!$item->getProduct()->getData($this->helper->getSpecialItemsAttribute())) {
                    continue;
                }
                $_product = $item->getProduct();
                $_sku = explode('-', $item->getSku());
                $skus[] = [
                    'sku'   => $_sku[0],
                    'size'  => $_sku[1],
                    'name'  => $_product->getName(),
                    'color' => $_product->getCstColore(),
                ];
            }

            // invio mail
            $templateVars = [
                'orderId' => $order->getIncrementId(),
                'bo_url'  => $this->backendUrlManager->getUrl('sales/order/view', ['order_id' => $order->getId()]),
                'skus'    => $skus,
            ];
            $this->helper->sendMail(self::DEFAULT_PVS_RECEIVED_SPECIAL_ITEMS_EMAIL_TEMPLATE_ID, $templateVars, $this->helper->getSpecialItemsEmail($order->getStoreId()));
        }
        $order->setStatus(self::ORDER_STATUS_CAN_PICK_UP_PVS);
        $order->addStatusToHistory(self::ORDER_STATUS_CAN_PICK_UP_PVS, "The order invoice was created, then PVS can pick up the order");
        try {
            $order->save();
        } catch (\Exception $e) {
            throw new \Exception("[PVS][SAVE_DOC_FILE_FTP] #{$order->getIncrementId()} save status error - {$e->getMessage()}");
        }

        $filenames = $this->getDocumentFilenamesFromCarrier($invoice);

        // creare pdf fattura con nome <n_ordine>.pdf
        $this->pdf->setInvoice($invoice);

        // caricarlo nell'ftp di pvs
        if (!$this->ftp->checkIfIsEnabled()) {
            return;
        }


        $dir = 'IN/';
        $flagReqInvoice = $order->getBillingAddress()->getRequestInvoice();


        try {
            // extra-cee
            if ($order->getShippingAddress()->getCountryId() != '' &&
                !in_array($order->getBillingAddress()->getCountryId(), $this->helper->getEuCountriesArray()) &&
                $filenames['free_export']) {
                // stampo la fattura pulita senza dichiarazione
                $this->pdf->cleanInvoice();

                // istruzioni reso e proforma reso a parte
                $path = $dir . $order->getIncrementId() . "-0.pdf";
                $template = $this->templateRepository->getById($this->helper->GetProformaRemTemplateId($order->getStoreId()));
                $this->pdf->setTemplate($template);
                $invoicePdf = $this->pdf->template2Pdf();
                $this->ftp->saveFileToFtp($path, $invoicePdf['filestream']);
            }
            // se è un ordine gift usare template senza totali
            if ($order->getGiftMessageId()) {
                if ($flagReqInvoice) {
                    $template = $this->templateRepository->getById($this->helper->GetNoTotalInvoiceTemplateId());
                } else {
                    $template = $this->templateRepository->getById($this->helper->GetNoTotalProformaTemplateId());
                }
            } else {
                if ($flagReqInvoice) {
                    $template = $this->templateRepository->getById($this->helper->GetInvoiceTemplateId($order->getStoreId()));
                } else {
                    $template = $this->templateRepository->getById($this->helper->GetProformaTemplateId($order->getStoreId()));
                }
            }
            $path = $dir . $filenames['document'];
            $this->pdf->setTemplate($template);
            $invoicePdf = $this->pdf->template2Pdf();
            $this->ftp->saveFileToFtp($path, $invoicePdf['filestream']);

            if ($filenames['free_export']) {
                // stampo la dichiarazione a parte
                $template = $this->templateRepository->getById($this->helper->GetDeclarationTemplateId($order->getStoreId()));
                $path = $dir . $filenames['free_export'];
                $this->pdf->setTemplate($template);
                $invoicePdf = $this->pdf->template2Pdf();
                $this->ftp->saveFileToFtp($path, $invoicePdf['filestream']);
            }
            if ($filenames['original_preference']) {
                $template = $this->templateRepository->getById($this->helper->GetDecPrefTemplateId($order->getStoreId()));
                $path = $dir . $filenames['original_preference'];
                $this->pdf->setTemplate($template);
                $invoicePdf = $this->pdf->template2Pdf();
                $this->ftp->saveFileToFtp($path, $invoicePdf['filestream']);
            }

        } catch (\Exception $e) {
            throw new \Exception("[PVS][SAVE_DOC_FILE_FTP] Error to save file {$path} - {$e->getMessage()}");
        }

        if (!$order->getGiftMessageId()) {
            return;
        }

        // gift message
        if (!$giftMessageTemplateId = $this->helper->GetGiftMessageTemplateId()) {
            return;
        }

        $orderPrefixPdfName = $order->getIncrementId() . '-0';
        $giftMessage = $this->messageFactory->create()->load($order->getGiftMessageId());
        $pathGift = $dir . $orderPrefixPdfName . '-COLOR.pdf';
        $this->pdf->setInvoice($invoice);
        $this->pdf->setGiftMessage($giftMessage);
        $templateGift = $this->templateRepository->getById($giftMessageTemplateId);
        $this->pdf->setTemplate($templateGift);
        $giftPdf = $this->pdf->template2Pdf();
        try {
            $this->ftp->saveFileToFtp($pathGift, $giftPdf['filestream']);
        } catch (\Exception $e) {
            throw new \Exception("[PVS][SAVE_DOC_FILE_FTP] Error to save file {$path} - {$e->getMessage()}");
        }
    }

    /**
     * @param $invoice
     * @throws \Exception
     */
    public function getDocumentFilenamesFromCarrier($invoice)
    {
        $order = $invoice->getOrder();
        $filename = $this->manageCarrierFiles();

        $carrier = null;
        $generalCarrier = null;
        $fileContent = $this->ftp->readFileFromFtp($filename);
        $data = $this->helper->getArrayFromCsv($fileContent);

        foreach ($data as $row) {

            if ($row['nazione'] == "*") {
                $generalCarrier = $row['vettore'];
                continue;
            }
            if ($row['nazione'] != $order->getShippingAddress()->getCountryId()) {
                continue;
            }
            $carrier = $row["vettore"];
        }

        if (!$carrier) {
            $carrier = $generalCarrier;
        }

        if (strpos($carrier, 'DHL') !== false) {
            $carrier = "DHL";
        }

        $docName = $order->getBillingAddress()->getRequestInvoice() ? "COMMERCIAL" : "PRO_FORMA";
        switch ($carrier) {
            case "UPS":
                return [
                    "document"            => $order->getIncrementId() . "-0_" . $docName . "_INVOICE_1.pdf",
                    "free_export"         => $order->getIncrementId() . "-0_DECLARATION_2.pdf",
                    "original_preference" => null,
                ];
            case "DHL":
            case "FEDEX":
                return [
                    "document"            => $order->getIncrementId() . "-0_" . $docName . "_INVOICE_1.pdf",
                    "free_export"         => null,
                    "original_preference" => null,
                ];
            default:
                return [
                    "document"            => $order->getIncrementId() . "-0.pdf",
                    "free_export"         => null,
                    "original_preference" => null,
                ];
        }

    }

    /**
     * @return mixed|null
     */
    public function manageCarrierFiles()
    {
        $globalName = "Vettori " . $this->helper->getCodDicl();
        $filePaths = $this->ftp->getFilesList(self::DEFAULT_PVS_COURIERS_DIR, $globalName);

        if (count($filePaths) == 1) {
            return $filePaths[0];
        }

        $file = null;
        $lastDate = null;

        try {
            foreach ($filePaths as $filePath) {
                $strFilename = $filePath;
                $strFilename = str_replace(".csv", "", $strFilename);
                $strFilename = str_replace(self::DEFAULT_PVS_COURIERS_DIR_NAME . "/", "", $strFilename);
                $year = substr($strFilename, 0, 4);
                $month = substr($strFilename, 4, 2);
                $day = substr($strFilename, 6, 2);
                $h = substr($strFilename, 8, 2);
                $m = substr($strFilename, 10, 2);

                $fileDate = "$year-$month-$day $h:$m";

                if (!$lastDate || strtotime($fileDate) > strtotime($lastDate)) {
                    $lastDate = $fileDate;
                    $file = $filePath;
                }
            }

            foreach ($filePaths as $filePath) {
                if ($filePath == $file) {
                    continue;
                }

                // move to old
                $filePathExplode = explode('/', $filePath);
                $this->ftp->moveFtpFile($filePath, self::DEFAULT_PVS_COURIERS_DIR . self::DEFAULT_PVS_COURIERS_OLD_DIR . end($filePathExplode));

            }

        } catch (\Exception $e) {
            throw new \Exception("[PVS][READ_DOC_FILE_FTP] Error to read file {$globalName} - {$e->getMessage()}");
        }


        return $file;
    }
}
