<?php
/**
 * @category    Magento 2
 * @copyright   Copyright (c) 2021 Drop (https://drop.it)
 * @author      Alberto Pallotto <a.pallotto@drop.it>
 */

namespace Drop\Pvs\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Encryption\EncryptorInterface;

/**
 * Class ConfigurationHelper
 * @package Drop\Pvs\Helper
 */
class ConfigurationHelper extends AbstractHelper
{
    const XML_SECTION = "drop_pvs";

    const XML_GROUP_GENERAL = self::XML_SECTION . "/general";
    const XML_PATH_ENABLED_MODULE = self::XML_GROUP_GENERAL . "/enabled";
    const XML_PATH_RMA_DAY_LIMIT_ENABLED_MODULE = self::XML_GROUP_GENERAL . "/enable_rma_day_limit";
    const XML_PATH_RMA_DAY_LIMIT_MODULE = self::XML_GROUP_GENERAL . "/rma_day_limit";

    const XML_GROUP_EMAILS = self::XML_SECTION . "/emails";
    const XML_PATH_ENABLED_EMAILS = self::XML_GROUP_EMAILS . "/enabled";
    const XML_PATH_CANCELED_ORDER_EMAILS = self::XML_GROUP_EMAILS . "/canceled_order";
    const XML_PATH_RMA_EMAILS = self::XML_GROUP_EMAILS . "/rma";

    const XML_GROUP_ORDERS = self::XML_SECTION . "/orders";
    const XML_PATH_ENABLED_ORDER_EXPORT = self::XML_GROUP_ORDERS . "/enabled_export";
    const XML_PATH_ORDER_EXPORT_STATUS = self::XML_GROUP_ORDERS . "/order_export_status";
    const XML_PATH_ORDER_SHIPPED_STATUS = self::XML_GROUP_ORDERS . "/shipped_status";
    const XML_PATH_ORDER_DELIVERED_STATUS = self::XML_GROUP_ORDERS . "/delivered_status";
    const XML_PATH_ORDER_INSURANCE = self::XML_GROUP_ORDERS . "/insurance_threshold";
    const XML_PATH_ORDER_COD_ENABLED = self::XML_GROUP_ORDERS . "/cod_enabled";
    const XML_PATH_ORDER_COD_CODE = self::XML_GROUP_ORDERS . "/cod_code";
    const XML_PATH_ORDER_TRANSLATION_COUNTRIES = self::XML_GROUP_ORDERS . "/translation_countries";

    const XML_GROUP_PRODUCTS = self::XML_SECTION . "/products";
    const XML_PATH_ENABLED_CONFIGURABLE_STOCK_RESET = self::XML_GROUP_PRODUCTS . "/configurable_stock_reset_enabled";
    const XML_PATH_EAN_ATTRIBUTE = self::XML_GROUP_PRODUCTS . "/ean_attribute";
    const XML_PATH_RESTOCK_FILE_FIELD_EAN = self::XML_GROUP_PRODUCTS . "/restock_file_ean_field";
    const XML_PATH_RESTOCK_FILE_FIELD_QTY = self::XML_GROUP_PRODUCTS . "/restock_file_qty_field";
    const XML_PATH_RESTOCK_FILE_FIELD_DDT_NUMBER = self::XML_GROUP_PRODUCTS . "/restock_file_ddt_number_field";
    const XML_PATH_RESTOCK_FILE_FIELD_DDT_DATE = self::XML_GROUP_PRODUCTS . "/restock_file_ddt_date_field";

    const XML_GROUP_FTP = self::XML_SECTION . "/ftp";
    const XML_PATH_ENABLED_FTP = self::XML_GROUP_FTP . "/enabled";
    const XML_PATH_ENABLED_SFTP = self::XML_GROUP_FTP . "/sftp";
    const XML_PATH_FTP_HOST = self::XML_GROUP_FTP . "/host";
    const XML_PATH_FTP_PORT = self::XML_GROUP_FTP . "/port";
    const XML_PATH_FTP_USER = self::XML_GROUP_FTP . "/username";
    const XML_PATH_FTP_PASSWORD = self::XML_GROUP_FTP . "/password";
    const XML_PATH_FTP_ROOT_PATH = self::XML_GROUP_FTP . "/root_dir";
    const XML_PATH_FTP_READ_DIR = self::XML_GROUP_FTP . "/read_dir";
    const XML_PATH_FTP_WRITE_DIR = self::XML_GROUP_FTP . "/write_dir";
    const XML_PATH_FTP_CARRIER_DIR = self::XML_GROUP_FTP . "/carrier_dir";
    const XML_PATH_FTP_RESTOCK_DIR = self::XML_GROUP_FTP . "/restock_dir";

    const XML_GROUP_COMMUNICATION = self::XML_SECTION . "/communications";
    const XML_PATH_ENABLED_COMMUNICATION = self::XML_GROUP_COMMUNICATION . "/enabled";
    const XML_PATH_COD_DICL = self::XML_GROUP_COMMUNICATION . "/cod_dicl";
    const XML_PATH_COD_SOCL = self::XML_GROUP_COMMUNICATION . "/cod_socl";
    const XML_PATH_COD_CLI_FOR = self::XML_GROUP_COMMUNICATION . "/cod_cli_for";

    const XML_GROUP_FILES = self::XML_SECTION . "/files";
    const XML_PATH_FILE_PRODUCTS = self::XML_GROUP_FILES . "/product";
    const XML_PATH_FILE_QTY_MOVEMENTS = self::XML_GROUP_FILES . "/qty_movements";
    const XML_PATH_FILE_STOCK = self::XML_GROUP_FILES . "/stock";
    const XML_PATH_FILE_ORDERS = self::XML_GROUP_FILES . "/order";
    const XML_PATH_FILE_ORDER_RESPONSE = self::XML_GROUP_FILES . "/order_response";
    const XML_PATH_FILE_SHIPMENTS = self::XML_GROUP_FILES . "/shipment";
    const XML_PATH_FILE_TRACKING = self::XML_GROUP_FILES . "/tracking";
    const XML_PATH_FILE_DELIVERED = self::XML_GROUP_FILES . "/delivered";
    const XML_PATH_FILE_RMA = self::XML_GROUP_FILES . "/rma";
    const XML_PATH_FILE_RMA_RESPONSE = self::XML_GROUP_FILES . "/rma_response";
    const XML_PATH_FILE_CARRIER = self::XML_GROUP_FILES . "/carrier";
    const XML_PATH_FILE_RESTOCK = self::XML_GROUP_FILES . "/restock";

    const XML_GROUP_PDF = self::XML_SECTION . "/pdf_invoice";
    const XML_PATH_PDF_ENABLED = self::XML_GROUP_PDF . "/pdf_enabled";
    const XML_PATH_PDF_URL = self::XML_GROUP_PDF . "/url";
    const XML_PATH_PDF_USERNAME = self::XML_GROUP_PDF . "/username";
    const XML_PATH_PDF_PASSWORD = self::XML_GROUP_PDF . "/password";
    const XML_PATH_PDF_LOGIN_ENDPOINT = self::XML_GROUP_PDF . "/login";
    const XML_PATH_PDF_INVOICE_ENDPOINT = self::XML_GROUP_PDF . "/invoice";
    const XML_PATH_PDF_RMA_PROFORMA_ENDPOINT = self::XML_GROUP_PDF . "/rma_proforma";

    const XML_GROUP_CSV = self::XML_SECTION . "/csv";
    const XML_PATH_CSV_ENCLOSURE = self::XML_GROUP_CSV . "/enclosure";
    const XML_PATH_CSV_SEPARATOR = self::XML_GROUP_CSV . "/separator";

    const XML_GROUP_GSPED = self::XML_SECTION . "/gsped_configuration";
    const XML_PATH_GSPED_ENABLED = self::XML_GROUP_GSPED . "/gsped_enabled";
    const XML_PATH_GSPED_API_URL = self::XML_GROUP_GSPED . "/api_url";
    const XML_PATH_GSPED_API_KEY = self::XML_GROUP_GSPED . "/api_key";
    const XML_PATH_GSPED_CLIENT_ID = self::XML_GROUP_GSPED . "/client_id";
    const XML_PATH_GSPED_RMA_CLIENT_ID = self::XML_GROUP_GSPED . "/rma_client_id";
    const XML_PATH_GSPED_CONNECTION_TIMEOUT = self::XML_GROUP_GSPED . "/connection_timeout";
    const XML_PATH_GSPED_TIMEOUT = self::XML_GROUP_GSPED . "/timeout";
    const XML_PATH_GSPED_READ_TIMEOUT = self::XML_GROUP_GSPED . "/read_timeout";
    const XML_PATH_GSPED_SAVE_LABEL = self::XML_GROUP_GSPED . "/save_label";

    const XML_GROUP_GSPED_CARRIER = self::XML_SECTION . "/gsped_carrier_info";
    const XML_PATH_GSPED_CARRIER_STANDARD_RMA_CARRIER_CODE = self::XML_GROUP_GSPED_CARRIER . "/standard_rma_carrier_code";
    const XML_PATH_GSPED_CARRIER_STANDARD_RMA_CARRIER_SERVICE = self::XML_GROUP_GSPED_CARRIER . "/standard_rma_carrier_service";
    const XML_PATH_GSPED_CARRIER_COUNTRIES_RMA = self::XML_GROUP_GSPED_CARRIER . "/country_rma_carriers";

    const XML_RMA_SETTINGS_GROUP = "sales/magento_rma";
    const XML_PATH_RMA_SETTINGS_NAME = self::XML_RMA_SETTINGS_GROUP . "/store_name";
    const XML_PATH_RMA_SETTINGS_STREET = self::XML_RMA_SETTINGS_GROUP . "/address";
    const XML_PATH_RMA_SETTINGS_STREET_2 = self::XML_RMA_SETTINGS_GROUP . "/address1";
    const XML_PATH_RMA_SETTINGS_CITY = self::XML_RMA_SETTINGS_GROUP . "/city";
    const XML_PATH_RMA_SETTINGS_PROVINCE = self::XML_RMA_SETTINGS_GROUP . "/region_id";
    const XML_PATH_RMA_SETTINGS_POSTCODE = self::XML_RMA_SETTINGS_GROUP . "/zip";
    const XML_PATH_RMA_SETTINGS_COUNTRY_ID = self::XML_RMA_SETTINGS_GROUP . "/country_id";
    const XML_PATH_RMA_SETTINGS_EMAIL = self::XML_RMA_SETTINGS_GROUP . "/email";
    const XML_PATH_RMA_SETTINGS_PHONE = self::XML_RMA_SETTINGS_GROUP . "/telephone";

    /**
     * @var EncryptorInterface
     */
    private $encryptor;

    /**
     * ConfigurationHelper constructor.
     * @param Context $context
     * @param EncryptorInterface $encryptor
     */
    public function __construct(
        Context $context,
        EncryptorInterface $encryptor
    )
    {
        parent::__construct($context);
        $this->encryptor = $encryptor;
    }

    /* GENERAL */
    /**
     * @return bool
     */
    public function isEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_MODULE);
    }

    /**
     * @return bool
     */
    public function isRmaDayLimitEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_RMA_DAY_LIMIT_ENABLED_MODULE);
    }

    /**
     * @return string
     */
    public function getRmaDayLimit(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_DAY_LIMIT_MODULE);
    }

    /* EMAIL */
    /**
     * @return bool
     */
    public function isEmailEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_EMAILS);
    }

    /**
     * @return string
     */
    public function getCanceledOrderEmail(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_CANCELED_ORDER_EMAILS);
    }

    /**
     * @return string
     */
    public function getRmaEmail(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_EMAILS);
    }

    /* ORDER */
    /**
     * @return bool
     */
    public function isOrderExportEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_ORDER_EXPORT);
    }

    /**
     * @return string
     */
    public function getOrderStatusToExport(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_ORDER_EXPORT_STATUS);
    }

    /**
     * @return string
     */
    public function getOrderShippedStatus(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_ORDER_SHIPPED_STATUS);
    }

    /**
     * @return string
     */
    public function getOrderDeliveredStatus(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_ORDER_DELIVERED_STATUS);
    }

    /**
     * @param null $storeId
     * @return float
     */
    public function getOrderInsuranceLimitValue($storeId = null): float
    {
        return (float)$this->scopeConfig->getValue(self::XML_PATH_ORDER_INSURANCE, 'store', $storeId);
    }

    /**
     * @return bool
     */
    public function isOrderPaymentCodEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ORDER_COD_ENABLED);
    }

    /**
     * @return string
     */
    public function getOrderPaymentCodCode(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_ORDER_COD_CODE);
    }

    /**
     * @return array
     */
    public function getOrderTranslationCountries(): array
    {
        return explode(',', $this->scopeConfig->getValue(self::XML_PATH_ORDER_TRANSLATION_COUNTRIES));
    }

    /* PRODUCT */
    /**
     * @return bool
     */
    public function isConfigurableResetStockEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_CONFIGURABLE_STOCK_RESET);
    }

    /**
     * @return string
     */
    public function getEanAttributeCode(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_EAN_ATTRIBUTE);
    }

    /**
     * @return string
     */
    public function getRestockFileEanField(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RESTOCK_FILE_FIELD_EAN);
    }

    /**
     * @return string
     */
    public function getRestockFileQtyField(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RESTOCK_FILE_FIELD_QTY);
    }

    /**
     * @return string
     */
    public function getRestockFileDdtNumberField(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RESTOCK_FILE_FIELD_DDT_NUMBER);
    }

    /**
     * @return string
     */
    public function getRestockFileDdtDateField(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RESTOCK_FILE_FIELD_DDT_DATE);
    }

    /* FTP */
    /**
     * @return bool
     */
    public function isFtpEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_FTP);
    }

    /**
     * @return bool
     */
    public function isSftpEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_SFTP);
    }

    /**
     * @return string
     */
    public function getFtpHost(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_HOST);
    }

    /**
     * @return string
     */
    public function getFtpPort(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_PORT);
    }

    /**
     * @return string
     */
    public function getFtpUsername(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_USER);
    }

    /**
     * @return string
     */
    public function getFtpPassword(): string
    {
        return $this->encryptor->decrypt((string)$this->scopeConfig->getValue(self::XML_PATH_FTP_PASSWORD));
    }

    /**
     * @return string
     */
    public function getFtpRootPath(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_ROOT_PATH);
    }

    /**
     * @return string
     */
    public function getFtpReadDir(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_READ_DIR);
    }

    /**
     * @return string
     */
    public function getFtpWriteDir(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_WRITE_DIR);
    }

    /**
     * @return string
     */
    public function getFtpCarrierDir(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_CARRIER_DIR);
    }

    /**
     * @return string
     */
    public function getFtpRestockDir(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FTP_RESTOCK_DIR);
    }

    /* COMMUNICATION */
    /**
     * @return bool
     */
    public function isCommunicationEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_ENABLED_COMMUNICATION);
    }

    /**
     * @return string
     */
    public function getCodDicl(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_COD_DICL);
    }

    /**
     * @return string
     */
    public function getCodSocl(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_COD_SOCL);
    }

    /**
     * @return string
     */
    public function getCodCliFor(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_COD_CLI_FOR);
    }

    /* FILES */
    /**
     * @return string
     */
    public function getProductFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_PRODUCTS);
    }

    /**
     * @return string
     */
    public function getQtyMovementsFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_QTY_MOVEMENTS);
    }

    /**
     * @return string
     */
    public function getStockFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_STOCK);
    }

    /**
     * @return string
     */
    public function getOrderFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_ORDERS);
    }

    /**
     * @return string
     */
    public function getOrderResponseFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_ORDER_RESPONSE);
    }

    /**
     * @return string
     */
    public function getShipmentFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_SHIPMENTS);
    }

    /**
     * @return string
     */
    public function getTrackingFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_TRACKING);
    }

    /**
     * @return string
     */
    public function getDeliveredFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_DELIVERED);
    }

    /**
     * @return string
     */
    public function getRmaFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_RMA);
    }

    /**
     * @return string
     */
    public function getRmaResponseFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_RMA_RESPONSE);
    }

    /**
     * @return string
     */
    public function getCarrierFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_CARRIER);
    }

    /**
     * @return string
     */
    public function getRestockFilename(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_FILE_RESTOCK);
    }

    /* PDF */
    /**
     * @return bool
     */
    public function isPdfEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_PDF_ENABLED);
    }

    /**
     * @return string
     */
    public function getPdfInvoiceUrl(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_PDF_URL);
    }

    /**
     * @return string
     */
    public function getPdfInvoiceUsername(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_PDF_USERNAME);
    }

    /**
     * @return string
     */
    public function getPdfInvoicePassword(): string
    {
        return $this->encryptor->decrypt((string)$this->scopeConfig->getValue(self::XML_PATH_PDF_PASSWORD));
    }

    /**
     * @return string
     */
    public function getPdfInoviceLoginEndpoint(): string
    {
        return $this->getPdfInvoiceUrl() . (string)$this->scopeConfig->getValue(self::XML_PATH_PDF_LOGIN_ENDPOINT);
    }

    /**
     * @return string
     */
    public function getPdfInvoiceDownloadEndpoint(): string
    {
        return $this->getPdfInvoiceUrl() . (string)$this->scopeConfig->getValue(self::XML_PATH_PDF_INVOICE_ENDPOINT);
    }

    /**
     * @return string
     */
    public function getPdfRmaProformaDownloadEndpoint(): string
    {
        return $this->getPdfInvoiceUrl() . (string)$this->scopeConfig->getValue(self::XML_PATH_PDF_RMA_PROFORMA_ENDPOINT);
    }

    /* CSV */
    /**
     * @return string
     */
    public function getCsvEnclosure(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_CSV_ENCLOSURE);
    }

    /**
     * @return string
     */
    public function getCsvSeparator(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_CSV_SEPARATOR);
    }

    /* GSPED CONFIGURATION */
    /**
     * @return bool
     */
    public function isGspedEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_GSPED_ENABLED);
    }

    /**
     * @return string
     */
    public function getGspedApiUrl(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_API_URL);
    }

    /**
     * @return string
     */
    public function getGspedApiKey(): string
    {
        return $this->encryptor->decrypt((string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_API_KEY));
    }

    /**
     * @return string
     */
    public function getGspedClientId(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_CLIENT_ID);
    }

    /**
     * @return string
     */
    public function getGspedRmaClientId(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_RMA_CLIENT_ID);
    }

    /**
     * @return string
     */
    public function getGspedConnectionTimeout(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_CONNECTION_TIMEOUT);
    }

    /**
     * @return string
     */
    public function getGspedTimeout(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_TIMEOUT);
    }

    /**
     * @return string
     */
    public function getGspedReadTimeout(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_READ_TIMEOUT);
    }

    /**
     * @return bool
     */
    public function isGspedSaveLabelEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(self::XML_PATH_GSPED_SAVE_LABEL);
    }

    /* GSPED CARRIER */
    /**
     * @return string
     */
    public function getGspedStandardRmaCarrierCode(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_CARRIER_STANDARD_RMA_CARRIER_CODE);
    }

    /**
     * @return string
     */
    public function getGspedStandardRmaCarrierService(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_GSPED_CARRIER_STANDARD_RMA_CARRIER_SERVICE);
    }

    /**
     * @param null $countryCode
     * @return array
     */
    public function getGspedCountryRmaCarrierService($countryCode = null): array
    {
        $data = (object)json_decode($this->scopeConfig->getValue(self::XML_PATH_GSPED_CARRIER_COUNTRIES_RMA));
        foreach ($data as $row) {
            ;
            if ($countryCode != $row->country_id) {
                continue;
            }

            return [$row->carrier_code, $row->carrier_service];
        }

        return [$this->getGspedStandardRmaCarrierCode(), $this->getGspedStandardRmaCarrierService()];
    }

    /* MAGENTO RMA SETTINGS */
    /**
     * @return string
     */
    public function getMageRmaName(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_NAME);
    }

    /**
     * @return string
     */
    public function getMageRmaStreet(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_STREET);
    }

    /**
     * @return string
     */
    public function getMageRmaStreet2(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_STREET_2);
    }

    /**
     * @return string
     */
    public function getMageRmaCity(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_CITY);
    }

    /**
     * @return string
     */
    public function getMageRmaRegionId(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_PROVINCE);
    }

    /**
     * @return string
     */
    public function getMageRmaPostcode(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_POSTCODE);
    }

    /**
     * @return string
     */
    public function getMageRmaCountryId(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_COUNTRY_ID);
    }

    /**
     * @return string
     */
    public function getMageRmaEmail(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_EMAIL);
    }

    /**
     * @return string
     */
    public function getMageRmaPhone(): string
    {
        return (string)$this->scopeConfig->getValue(self::XML_PATH_RMA_SETTINGS_PHONE);
    }
}

