<?php


namespace Drop\Pvs\Plugin;


use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Logger\Logger;
use Magento\Framework\Exception\LocalizedException;
use Magento\Rma\Helper\Data;
use Magento\Sales\Api\OrderRepositoryInterface;

class RmaDataHelper
{
    protected $configuration;
    protected $orderRepository;
    protected $logger;

    /**
     * RmaDataHelper constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param OrderRepositoryInterface $orderRepository
     * @param Logger $logger
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        OrderRepositoryInterface $orderRepository,
        Logger $logger
    )
    {
        $this->configuration = $configurationHelper;
        $this->orderRepository = $orderRepository;
        $this->logger = $logger;
    }

    /**
     * @param Data $subject
     * @param $result
     * @param $order
     * @return false|mixed|void
     * @throws LocalizedException
     */
    public function afterCanCreateRma(Data $subject, $result, $order)
    {
        // se non è attivo pvs
        // se non è attivo il controllo dei giorni
        // se magento nega la possibilità
        // ==> torno il risultato di magento
        if (
            !$this->configuration->isEnabled()
            || !$this->configuration->isRmaDayLimitEnabled()
            || !$result
        ) {
            return $result;
        }

        if (!is_object($order)) {
            $order = $this->orderRepository->get($order);
        }

        if (!$order) {
            $this->logger->error("[CAN CREATE RMA] error, can't find order");
            throw new LocalizedException(__('Please check this order for errors.'));
            return;
        }

        // gestione ordini precedenti all'introduzione di pvs
        $deliveryDate = null;
        if (
            strtotime($order->getCreatedAt()) < strtotime($this->configuration->getActiveFromDate())
            && !$order->getDropPvsDeliveryDate()
        ) {
            $deliveryDate = $order->getCreatedAt();
        }

        if (!$order->getDropPvsDeliveryDate() && !$deliveryDate) {
            return false;
        }

        return $this->checkDateDifference($deliveryDate, $result);
    }

    public function checkDateDifference($deliveryDate, $result)
    {
        $difference = $this->getWorkingDays($deliveryDate);

        if ($difference > (int)$this->configuration->getRmaDayLimit()) {
            return false;
        }

        return $result;
    }

    /**
     * @param $from
     * @param $to
     * @return int
     * @throws \Exception
     */
    public function getWorkingDays($from, $to = null)
    {
        $workingDays = [1, 2, 3, 4, 5];
        $holidayDays = ['*-01-01', '*-01-06', '*-04-25', '*-05-01', '*-06-02', '*-08-15', '*-11-01', '*-12-08', '*-12-25', '*-12-26'];
        // manca la Pasqua che è un giorno variabile...
        $from = new \DateTime($from);
        $to = $to ? new \DateTime($to) : new \DateTime(date('Y-m-d'));
        $to->modify('+1 day');
        $interval = new \DateInterval('P1D');
        $periods = new \DatePeriod($from, $interval, $to);

        $days = 0;
        foreach ($periods as $period) {
            if (!in_array($period->format('N'), $workingDays)) continue;
            if (in_array($period->format('Y-m-d'), $holidayDays)) continue;
            if (in_array($period->format('*-m-d'), $holidayDays)) continue;
            $days++;
        }
        return $days;
    }
}
