<?php


namespace Drop\Pvs\Model\Consumer\Export;


use Drop\Pvs\Api\FileInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Downloader;
use Drop\Pvs\Model\Ftp\Modifier;
use Drop\Pvs\Model\Ftp\Uploader;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;

class Restock
{
    const EMAIL_TEMPLATE = "drop_pvs_restock_errors_email_template";

    private $configuration;
    private $utilities;
    private $logger;
    private $downloader;
    private $modifier;
    private $uploader;
    private $productsRepositoryInterface;
    private $searchCriteriaBuilder;

    public function __construct(
        ConfigurationHelper        $configurationHelper,
        UtilitiesHelper            $utilitiesHelper,
        Logger                     $logger,
        Downloader                 $downloader,
        Modifier                   $modifier,
        Uploader                   $uploader,
        ProductRepositoryInterface $productsRepositoryInterface,
        SearchCriteriaBuilder      $searchCriteriaBuilder
    )
    {
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->logger = $logger;
        $this->downloader = $downloader;
        $this->modifier = $modifier;
        $this->uploader = $uploader;
        $this->productsRepositoryInterface = $productsRepositoryInterface;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @param FileInterface $data
     * @return bool
     */
    public function process(FileInterface $data)
    {
        $this->logger->info("Exporting Restock");

        // genero XDOCE
        $this->logger->info("Generate XDOCE [Restock]");

        foreach ($data->getFiles() as $file) {
            $restockData = $this->generateData($file);
            
            if (count($restockData)) {
                $this->saveFiles($restockData, $data);
            }

            $this->manageFtpFile($file, $data->getCanDeleteFiles());
        }

        return true;
    }

    /**
     * @param $file
     * @return array
     */
    public function generateData($file): array
    {
        $rowsData = [];
        $filename = explode("/", $file);
        $filenameText = array_pop($filename);
        $rows = $this->utilities->getAssocArrayFromFile($this->configuration->getFtpRestockDir(), $filenameText);
        foreach ($rows as $row) {
            if (count($row) < 4) {
                // serve per gestire la riga vuota che i csv hanno alla fine del file
                continue;
            }
            if (!isset($row[$this->configuration->getRestockFileQtyField()]) && count($row) >= 4) {
                $this->notifyError($filenameText, "field {$this->configuration->getRestockFileQtyField()} not set");
                break;
            }
            if (trim($row[$this->configuration->getRestockFileQtyField()]) == "") {
                // in questo caso significa che la riga contiene tutti i campi, ma sono vuoti (caso successo),
                // quindi salto senza notificare
                continue;
            }

            $this->searchCriteriaBuilder->addFilter($this->configuration->getEanAttributeCode(), trim($row[$this->configuration->getRestockFileEanField()]));
            if (!$product = $this->productsRepositoryInterface->getList($this->searchCriteriaBuilder->create())->getItems()) {
                $this->notifyError($filenameText, "item {$row[$this->configuration->getRestockFileEanField()]} not found");
                continue;
            }
            $product = reset($product);

            $rowsData[] = [
                'COD_SOCL'    => $this->configuration->getCodSocl(),
                'COD_DICL'    => $this->configuration->getCodDicl(),
                'COD_ART'     => $product->getSku(),
                'QTA_RIGA'    => trim($row[$this->configuration->getRestockFileQtyField()]),
                'COD_UM'      => 'PZ',
                'COD_CMOV'    => 'CAR',
                'COD_CLI_FOR' => $this->configuration->getCodCliFor(),
                'BOLLA'       => trim($row[$this->configuration->getRestockFileDdtNumberField()]),
                'DATA_BOLLA'  => trim($row[$this->configuration->getRestockFileDdtDateField()]),
            ];
        }

        return $rowsData;
    }

    /**
     * @param $filename
     * @param $canDeleteFile
     */
    protected function manageFtpFile($filename, $canDeleteFile)
    {
        if ($canDeleteFile) {
            $this->logger->info("delete file - {$filename}");
            $this->modifier->delete($filename);
            return;
        }

        $this->logger->info("move file to DONE - {$filename}");
        $this->modifier->moveToDone($filename);
    }

    /**
     * @param $content
     * @param $data
     */
    public function saveFiles($content, $data)
    {
        // gestione file caricato dal cliente
        //se $canDeleteFiles a true, elimino il file, sennò lo sposto sulla sotto cartella!
        if ($data->getCanDeleteFiles()) {
            foreach ($data->getFiles() as $file) {
                $this->modifier->delete($file);
            }
        } else {
            foreach ($data->getFiles() as $file) {
                $this->modifier->moveToOld($file);
            }
        }


        $filename = $this->utilities->completeFilenameWithDateTime($this->configuration->getQtyMovementsFilename());
        $this->uploader->execute($filename, $content);
    }

    /**
     * @param $filename
     * @param $message
     */
    protected function notifyError($filename, $message)
    {
        $to = $this->configuration->getRestockErrorsEmail();
        if (!$this->configuration->isEmailEnabled() || !is_array($to) || !count($to)) {
            return;
        }

        $templateVars = [
            "filename" => $filename,
            "message"  => $message,
        ];
        $this->utilities->sendMail(self::EMAIL_TEMPLATE, $templateVars, $to);
    }
}
