<?php


namespace Drop\Pvs\Console\Command\Import;


use Drop\Pvs\Cron\Import\DeletedOrder as Cron;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Model\Consumer\Import\DeletedOrder as Consumer;
use Drop\Pvs\Model\Ftp\Downloader;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\MessageQueue\PublisherInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class DeletedOrder extends Command
{
    const FILENAME = 'filename';
    const DELETE = 'delete';
    const CREATE_JOB = 'create-job';

    private $configuration;
    private $publisher;
    private $downloader;
    private $appState;
    private $consumer;
    private $cron;

    /**
     * Shipment constructor.
     * @param ConfigurationHelper $configuration
     * @param PublisherInterface $publisher
     * @param Downloader $downloader
     * @param State $appState
     * @param Consumer $consumer
     * @param Cron $cron
     */
    public function __construct(
        ConfigurationHelper $configuration,
        PublisherInterface $publisher,
        Downloader $downloader,
        State $appState,
        Consumer $consumer,
        Cron $cron
    )
    {
        $this->configuration = $configuration;
        $this->publisher = $publisher;
        $this->downloader = $downloader;
        $this->appState = $appState;
        $this->consumer = $consumer;
        $this->cron = $cron;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::FILENAME,
                null,
                InputOption::VALUE_OPTIONAL,
                'get filename'
            ),
            new InputOption(
                self::DELETE,
                null,
                InputOption::VALUE_NONE,
                'if you want to delete files'
            ),
            new InputOption(
                self::CREATE_JOB,
                null,
                InputOption::VALUE_NONE,
                'if you want to create job'
            ),
        ];
        $this->setName('drop:pvs:import:deleted-order');
        $this->setDescription('Manual deleted order import from Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\FileSystemException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $filename = $input->getOption(self::FILENAME) ?? $this->configuration->getDeletedOrderFilename();
        $canDelete = $input->getOption(self::DELETE) ?? false;
        $createJob = $input->getOption(self::CREATE_JOB) ?? false;

        $files = $this->downloader->list($filename, $this->configuration->getFtpReadDir());

        $data = $this->cron->getPublisherData($files, !$canDelete);

        if (!$createJob) {
            $this->consumer->process($data);
            return;
        }

        $this->publisher->publish(Cron::TOPIC_NAME, $data);
    }
}
