<?php


namespace Drop\Pvs\Cron\Export;


use Drop\Pvs\Api\Export\ProductsInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Consumer\Export\Products as Consumer;
use Drop\Pvs\Model\Publisher;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\Search\FilterGroupBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Bulk\BulkManagementInterface;
use Magento\Framework\Bulk\OperationInterface;
use Magento\Framework\DataObject\IdentityGeneratorInterface;
use Magento\Framework\MessageQueue\PublisherInterface;
use Magento\Framework\Serialize\SerializerInterface;

class Products
{
    const TOPIC_NAME = 'pvs.export.products';

    private $publisher;
    private $consumer;
    private $configuration;
    private $searchCriteriaBuilder;
    private $productRepository;
    private $productsInterface;
    private $filter;
    private $filterGroup;

    /**
     * Products constructor.
     * @param Publisher $publisher
     * @param Consumer $consumer
     * @param ConfigurationHelper $configurationHelper
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param ProductRepositoryInterface $productRepository
     * @param ProductsInterface $productsInterface
     * @param FilterBuilder $filter
     * @param FilterGroupBuilder $filterGroup
     */
    public function __construct(
        Publisher $publisher,
        Consumer $consumer,
        ConfigurationHelper $configurationHelper,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ProductRepositoryInterface $productRepository,
        ProductsInterface $productsInterface,
        FilterBuilder $filter,
        FilterGroupBuilder $filterGroup
    )
    {
        $this->publisher = $publisher;
        $this->consumer = $consumer;
        $this->configuration = $configurationHelper;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->productRepository = $productRepository;
        $this->productsInterface = $productsInterface;
        $this->filter = $filter;
        $this->filterGroup = $filterGroup;
    }

    public function execute()
    {
        if (!$this->configuration->isEnabled() || !$this->configuration->isCommunicationEnabled()) {
            return;
        }

        $filter1 = $this->filter->setField("is_pvs_exported")
            ->setValue(0)
            ->setConditionType('eq')->create();
        $filter2 = $this->filter->setField("is_pvs_exported")
            ->setConditionType('null')
            ->create();
        $filterGroup1 = $this->filterGroup
            ->addFilter($filter1)
            ->addFilter($filter2)
            ->create();
        $this->searchCriteriaBuilder->setFilterGroups([$filterGroup1]);
        $this->searchCriteriaBuilder->addFilter('type_id', 'simple');
        $searchCriteria = $this->searchCriteriaBuilder->create();
        if (!$products = $this->productRepository->getList($searchCriteria)->getItems()) {
            return;
        }

        $publisherData = $this->getPublisherData($products);

        if (!$this->configuration->isJobEnabled()) {
            $this->consumer->process($publisherData);
            return;
        }

        $this->publisher->publish(self::TOPIC_NAME, $publisherData, "Export Products");
    }

    /**
     * @param $products
     * @param bool $saveLocal
     * @param bool $saveToFtp
     * @param bool $forced
     * @return ProductsInterface
     */
    public function getPublisherData($products, bool $saveLocal = false, bool $saveToFtp = true, bool $forced = false): ProductsInterface
    {
        $productsIds = [];
        foreach ($products as $product) {
            $productsIds[] = $product->getEntityId();
        }

        $this->productsInterface->setProductsIds($productsIds);
        $this->productsInterface->setSaveLocal($saveLocal);
        $this->productsInterface->setSaveToFtp($saveToFtp);
        $this->productsInterface->setIsForced($forced);

        return $this->productsInterface;
    }
}
