<?php


namespace Drop\Pvs\Console\Command\Export;

use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Model\Ftp\Downloader;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\App\State;
use Magento\Framework\App\Area;
use Magento\Rma\Api\RmaRepositoryInterface;
use Magento\Framework\MessageQueue\PublisherInterface;
use \Drop\Pvs\Cron\Export\Restock as ExportRestock;
use \Drop\Pvs\Model\Consumer\Export\Restock as ExportRestockConsumer;

class Restock extends Command
{
    const FILENAME = 'filename';
    const DELETE = 'delete';
    const CREATE_JOB = 'create-job';

    /**
     * @var ConfigurationHelper
     */
    private $configuration;
    /**
     * @var State
     */
    private $appState;
    /**
     * @var PublisherInterface
     */
    private $publisher;
    /**
     * @var Downloader
     */
    private $downloader;
    /**
     * @var ExportRestock
     */
    private $exportRestock;
    /**
     * @var ExportRestockConsumer
     */
    private $exportRestockConsumer;


    public function __construct(
        ConfigurationHelper $configurationHelper,
        PublisherInterface $publisher,
        State $appState,
        Downloader $downloader,
        ExportRestock $exportRestock,
        ExportRestockConsumer $exportRestockConsumer
    )
    {
        $this->configuration = $configurationHelper;
        $this->appState = $appState;
        $this->publisher = $publisher;
        $this->downloader = $downloader;
        $this->exportRestock = $exportRestock;
        $this->exportRestockConsumer = $exportRestockConsumer;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::FILENAME,
                null,
                InputOption::VALUE_REQUIRED,
                'get filename'
            ),
            new InputOption(
                self::DELETE,
                null,
                InputOption::VALUE_NONE,
                'if you want to delete ftp file'
            ),
            new InputOption(
                self::CREATE_JOB,
                null,
                InputOption::VALUE_NONE,
                'if you want to create job'
            ),
        ];
        $this->setName('drop:pvs:export:restock');
        $this->setDescription('Manual restock export to Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $filename = $input->getOption(self::FILENAME) ?? $this->configuration->getRestockFilename();
        $canDelete = $input->getOption(self::DELETE) ?? false;
        $createJob = $input->getOption(self::CREATE_JOB) ?? false;

        if (!$files = $this->downloader->list($filename, $this->configuration->getFtpRestockDir())) {
            $output->writeln('<error>No $files was found.</error>');
            return;
        }

        $data = $this->exportRestock->getPublisherData($files, $canDelete);
        if (!$createJob) {
            $this->exportRestockConsumer->process($data);
            return;
        }

        $this->publisher->publish(ExportRestock::TOPIC_NAME, $data);
    }
}
