<?php

namespace Drop\Pvs\Console\Command\Export;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\MessageQueue\PublisherInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use \Drop\Pvs\Cron\Export\Products as ExportProducts;
use \Drop\Pvs\Model\Consumer\Export\Products as ExportProductsConsumer;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\Api\Search\FilterGroupBuilder;

class Products extends Command
{
    const SKUS = 'skus';
    const SAVE_LOCAL = 'local';
    const SAVE_TO_FTP = 'ftp';
    const FORCED = 'forced';
    const CREATE_JOB = 'create-job';

    private $productRepository;
    private $searchCriteriaBuilder;
    private $exportProducts;
    private $publisher;
    private $appState;
    private $exportProductsConsumer;
    private $filter;
    private $filterGroup;

    /**
     * Products constructor.
     * @param ProductRepositoryInterface $productRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param ExportProducts $exportProducts
     * @param PublisherInterface $publisher
     * @param State $appState
     * @param ExportProductsConsumer $exportProductsConsumer
     * @param FilterBuilder $filter
     * @param FilterGroupBuilder $filterGroup
     */
    public function __construct(
        ProductRepositoryInterface $productRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ExportProducts $exportProducts,
        PublisherInterface $publisher,
        State $appState,
        ExportProductsConsumer $exportProductsConsumer,
        FilterBuilder $filter,
        FilterGroupBuilder $filterGroup
    )
    {
        $this->productRepository = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->exportProducts = $exportProducts;
        $this->publisher = $publisher;
        $this->appState = $appState;
        $this->exportProductsConsumer = $exportProductsConsumer;
        $this->filter = $filter;
        $this->filterGroup = $filterGroup;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::SKUS,
                null,
                InputOption::VALUE_OPTIONAL,
                'get products sku separated with ,'
            ),
            new InputOption(
                self::SAVE_LOCAL,
                null,
                InputOption::VALUE_NONE,
                'if you want to save in local folder'
            ),
            new InputOption(
                self::SAVE_TO_FTP,
                null,
                InputOption::VALUE_NONE,
                'if you want to save on remote ftp folder'
            ),
            new InputOption(
                self::FORCED,
                null,
                InputOption::VALUE_NONE,
                'if you want to force export'
            ),
            new InputOption(
                self::CREATE_JOB,
                null,
                InputOption::VALUE_NONE,
                'if you want to create job'
            ),
        ];
        $this->setName('drop:pvs:export:products');
        $this->setDescription('Manual products export to Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $skus = $input->getOption(self::SKUS) ?? false;
        $saveLocal = $input->getOption(self::SAVE_LOCAL) ?? false;
        $saveToFtp = $input->getOption(self::SAVE_TO_FTP) ?? false;
        $forced = $input->getOption(self::FORCED) ?? false;
        $createJob = $input->getOption(self::CREATE_JOB) ?? false;

        if (!$forced) {
            $filter1 = $this->filter->setField("is_pvs_exported")
                ->setValue(0)
                ->setConditionType('eq')->create();
            $filter2 = $this->filter->setField("is_pvs_exported")
                ->setConditionType('null')
                ->create();
            $filterGroup1 = $this->filterGroup
                ->addFilter($filter1)
                ->addFilter($filter2)
                ->create();
            $this->searchCriteriaBuilder->setFilterGroups([$filterGroup1]);
        }
        $this->searchCriteriaBuilder->addFilter('type_id', 'simple');
        if ($skus) {
            $this->searchCriteriaBuilder->addFilter('sku', $skus, 'in');
        }
        $searchCriteria = $this->searchCriteriaBuilder->create();

        if (!$products = $this->productRepository->getList($searchCriteria)->getItems()) {
            $output->writeln('<error>No products was found.</error>');
            return;
        }

        $data = $this->exportProducts->getPublisherData($products, $saveLocal, $saveToFtp, $forced);

        if (!$createJob) {
            $this->exportProductsConsumer->process($data);
            return;
        }

        $this->publisher->publish(ExportProducts::TOPIC_NAME, $data);
    }
}
