<?php


namespace Drop\Pvs\Model\Consumer\Import;


use Drop\Pvs\Api\FileInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Downloader;
use Drop\Pvs\Model\Ftp\Modifier;
use Drop\Pvs\Model\Stock\ConfigurableAlignment;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\Io\File;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemsSaveInterface;
use Drop\Pvs\Model\Mail\Template\TransportBuilder;

class Stock
{
    const MAGENTO_TABLE_SALES_ORDER = 'sales_order';
    const MAGENTO_TABLE_SALES_ORDER_ITEM = 'sales_order_item';
    const EAV_ATTRIBUTE = 'eav_attribute';
    const EAV_ATTRIBUTE_VALUE = 'catalog_product_entity_varchar';
    const PRODUCT = 'catalog_product_entity';

    private $logger;
    private $configuration;
    private $utilities;
    private $modifier;
    private $searchCriteriaBuilder;
    private $productRepository;
    private $sourceItem;
    private $sourceItemsSave;
    private $resourceConnection;
    private $reservedQtys;
    private $configurableAlignment;
    private $transportBuilder;

    /**
     * Stock constructor.
     * @param Logger $logger
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param Modifier $modifier
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param ProductRepositoryInterface $productRepository
     * @param SourceItemInterface $sourceItem
     * @param SourceItemsSaveInterface $sourceItemsSave
     * @param ResourceConnection $resourceConnection
     * @param ConfigurableAlignment $configurableAlignment
     * @param TransportBuilder $transportBuilder
     */
    public function __construct(
        Logger $logger,
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        Modifier $modifier,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ProductRepositoryInterface $productRepository,
        SourceItemInterface $sourceItem,
        SourceItemsSaveInterface $sourceItemsSave,
        ResourceConnection $resourceConnection,
        ConfigurableAlignment $configurableAlignment,
        TransportBuilder $transportBuilder
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->modifier = $modifier;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->productRepository = $productRepository;
        $this->sourceItem = $sourceItem;
        $this->sourceItemsSave = $sourceItemsSave;
        $this->resourceConnection = $resourceConnection;
        $this->configurableAlignment = $configurableAlignment;
        $this->transportBuilder = $transportBuilder;
    }

    /**
     * @param FileInterface $data
     */
    public function process(FileInterface $data)
    {
        if (!count($data->getFiles())) {
            return;
        }

        $summaryStock = [];

        foreach ($data->getFiles() as $stockFile) {
            $sourceItems = [];
            $expl = explode("/", $stockFile);
            $filename = array_pop($expl);
            $csvData = $this->utilities->getAssocArrayFromFile(implode("/", $expl), $filename);

            foreach ($csvData as $row) {
                if (!isset($row['COD_MCON']) || $row["COD_MCON"] != "1") {
                    continue;
                }

                $this->searchCriteriaBuilder->addFilter("sku", $row["COD_ART"]);
                foreach ($this->productRepository->getList($this->searchCriteriaBuilder->create())->getItems() as $product) {
                    $this->sourceItem->setSourceCode("default");
                    $this->sourceItem->setSku($product->getSku());
                    $quantity = $this->calcSalableQty($row["QTA_DISP"], $row["COD_ART"]);
                    $this->sourceItem->setStatus($quantity > 0 ? 1 : 0);
                    $this->sourceItem->setQuantity($quantity);
                    $sourceItems[] = $this->sourceItem;

                    // loggo differenza fra disponibile e presente
                    if ($quantity <> $row["QTA_DISP"]) {
                        $this->logger->info("Product {$row["COD_ART"]} pvs stock was {$row["QTA_DISP"]} - imported {$quantity}");
                    }
                }
                $this->logger->info("{$row["COD_ART"]}: qta passata {$row["QTA_DISP"]} - qta calcolata/salvata {$quantity}");
                $summaryStock[] = "{$row["COD_ART"]}: qta passata {$row["QTA_DISP"]} - qta calcolata/salvata {$quantity}";

                $this->sourceItemsSave->execute([$this->sourceItem]);
            }

//            $this->sourceItemsSave->execute($sourceItems);

            $this->manageFtpFile($stockFile, $data->getCanDeleteFiles());
        }

        if ($this->configuration->isConfigurableResetStockEnabled()) {
            $this->configurableAlignment->process();
        }

        $this->sendLogMail($summaryStock);
    }

    /**
     * @param $stock
     * @param $sku
     * @return int
     */
    protected function calcSalableQty($stock, $sku): int
    {
        if (!$this->reservedQtys) {
            $this->reservedQtys = $this->getSalableProductQty();
        }
        if ($stock == 0) {
            return 0;
        }
        foreach ($this->reservedQtys as $reservedQty) {
            if ($reservedQty["sku"] !== $sku) {
                continue;
            }

            return ($reservedQty["sku_ordered"] < $stock) ? (int)($stock - $reservedQty["sku_ordered"]) : 0;
        }

        return $stock;
    }

    /**
     * @return array
     */
    protected function getSalableProductQty(): array
    {
        $intervalDays = 10;

        $select = $this->resourceConnection->getConnection()
            ->select()
            ->from(
                ['so' => self::MAGENTO_TABLE_SALES_ORDER],
                ['soi.qty_ordered', new \Zend_Db_Expr('sum(qty_ordered) as sku_ordered')]
            )
            ->joinLeft(['soi' => self::MAGENTO_TABLE_SALES_ORDER_ITEM], "soi.order_id = so.entity_id")
            ->joinLeft(['eav' => self::EAV_ATTRIBUTE], "attribute_code = '{$this->configuration->getEanAttributeCode()}'")
            ->joinLeft(['product' => self::PRODUCT], "product.sku = soi.sku")
            ->joinLeft(['sku' => self::EAV_ATTRIBUTE_VALUE],
                "eav.attribute_id = sku.attribute_id and sku.row_id = product.row_id")
            ->where('soi.product_type=?', "simple")
            ->where('so.status IN(?)', ['processing', 'pending', 'can_ship'])
            ->where('so.state IN(?)', ['processing', 'new'])
            ->where('so.created_at >?', [new \Zend_Db_Expr("CURDATE() - INTERVAL {$intervalDays} DAY")])
            ->group(["sku.value"]);

        return $this->resourceConnection->getConnection()->fetchAll($select);
    }

    /**
     * @param $filename
     * @param $canDeleteFile
     */
    protected function manageFtpFile($filename, $canDeleteFile)
    {
        if ($canDeleteFile) {
            $this->logger->info("delete file - {$filename}");
            $this->modifier->delete($filename);
            return;
        }

        $this->logger->info("move file to DONE - {$filename}");
        $this->modifier->moveToDone($filename);
    }

    protected function sendLogMail($summaryData)
    {
        if (
            !$this->configuration->isStockSummaryEnabled() ||
            !count($this->configuration->getStockSummaryEmails())
        ) {
            return;
        }

        $this->transportBuilder->setTemplateIdentifier("drop_pvs_stock_log")
            ->setTemplateOptions(['area' => \Magento\Framework\App\Area::AREA_FRONTEND, 'store' => 1])
            ->setTemplateVars(
                ['skus_html' => implode("<br>", $summaryData)]
            )
            ->setFromByScope($this->configuration->getFromEmail());
        foreach ($this->configuration->getStockSummaryEmails() as $stockSummaryEmail) {
            $this->transportBuilder->addTo($stockSummaryEmail);
        }
        try {
            $this->transportBuilder->getTransport()->sendMessage();
        } catch (Exception $e) {
            echo $e->getMessage();
            exit;
        }
    }
}
