<?php


namespace Drop\Pvs\Console\Command\Import;


use Drop\Pvs\Cron\Import\Shipment as ImportShipmentCron;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Model\Consumer\Import\Shipment as ImportShipmentConsumer;
use Drop\Pvs\Model\Ftp\Downloader;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\MessageQueue\PublisherInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class Shipment extends Command
{
    const FILENAME = 'filename';
    const DELETE = 'delete';
    const CREATE_JOB = 'create-job';

    private $configuration;
    private $publisher;
    private $downloader;
    private $appState;
    private $importShipmentConsumer;
    private $importShipmentCron;

    /**
     * Shipment constructor.
     * @param ConfigurationHelper $configuration
     * @param PublisherInterface $publisher
     * @param Downloader $downloader
     * @param State $appState
     * @param ImportShipmentConsumer $importShipmentConsumer
     * @param ImportShipmentCron $importShipmentCron
     */
    public function __construct(
        ConfigurationHelper $configuration,
        PublisherInterface $publisher,
        Downloader $downloader,
        State $appState,
        ImportShipmentConsumer $importShipmentConsumer,
        ImportShipmentCron $importShipmentCron
    )
    {
        $this->configuration = $configuration;
        $this->publisher = $publisher;
        $this->downloader = $downloader;
        $this->appState = $appState;
        $this->importShipmentConsumer = $importShipmentConsumer;
        $this->importShipmentCron = $importShipmentCron;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::FILENAME,
                null,
                InputOption::VALUE_OPTIONAL,
                'get filename'
            ),
            new InputOption(
                self::DELETE,
                null,
                InputOption::VALUE_NONE,
                'if you want to delete files'
            ),
            new InputOption(
                self::CREATE_JOB,
                null,
                InputOption::VALUE_NONE,
                'if you want to create job'
            ),
        ];
        $this->setName('drop:pvs:import:shipment');
        $this->setDescription('Manual shipment import from Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\FileSystemException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $filename = $input->getOption(self::FILENAME) ?? $this->configuration->getShipmentFilename();
        $canDelete = $input->getOption(self::DELETE) ?? false;
        $createJob = $input->getOption(self::CREATE_JOB) ?? false;

        $files = $this->downloader->list($filename, $this->configuration->getFtpReadDir());

        $data = $this->importShipmentCron->getPublisherData($files, !$canDelete);

        if (!$createJob) {
            $this->importShipmentConsumer->process($data);
            return;
        }

        $this->publisher->publish(ImportShipmentCron::TOPIC_NAME, $data);
    }
}
