<?php


namespace Drop\Pvs\Console\Command\Export;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\App\State;
use Magento\Framework\App\Area;
use Magento\Rma\Api\RmaRepositoryInterface;
use Magento\Framework\MessageQueue\PublisherInterface;
use \Drop\Pvs\Cron\Export\Rma as ExportRma;
use \Drop\Pvs\Model\Consumer\Export\Rma as ExportRmaConsumer;

class Rma extends Command
{
    const RMA_ENTITY_ID = 'entity-id';
    const SAVE_LOCAL = 'local';
    const SAVE_TO_FTP = 'ftp';
    const FORCED = 'forced';
    const CREATE_JOB = 'create-job';

    /**
     * @var State
     */
    private $appState;
    /**
     * @var RmaRepositoryInterface
     */
    private $rmaRepository;
    /**
     * @var PublisherInterface
     */
    private $publisher;
    /**
     * @var ExportRma
     */
    private $jobExportRma;
    /**
     * @var ExportRmaConsumer
     */
    private $exportRmaConsumer;

    /**
     * Rma constructor.
     * @param RmaRepositoryInterface $rmaRepository
     * @param PublisherInterface $publisher
     * @param ExportRma $jobExportRma
     * @param State $appState
     * @param ExportRmaConsumer $exportRmaConsumer
     */
    public function __construct(
        RmaRepositoryInterface $rmaRepository,
        PublisherInterface $publisher,
        ExportRma $jobExportRma,
        State $appState,
        ExportRmaConsumer $exportRmaConsumer
    )
    {
        $this->appState = $appState;
        $this->rmaRepository = $rmaRepository;
        $this->publisher = $publisher;
        $this->jobExportRma = $jobExportRma;
        $this->exportRmaConsumer = $exportRmaConsumer;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::RMA_ENTITY_ID,
                null,
                InputOption::VALUE_REQUIRED,
                'get rma entity id'
            ),
            new InputOption(
                self::SAVE_LOCAL,
                null,
                InputOption::VALUE_NONE,
                'if you want to save in local folder'
            ),
            new InputOption(
                self::SAVE_TO_FTP,
                null,
                InputOption::VALUE_NONE,
                'if you want to save on remote ftp folder'
            ),
            new InputOption(
                self::FORCED,
                null,
                InputOption::VALUE_NONE,
                'if you want to force export'
            ),
            new InputOption(
                self::CREATE_JOB,
                null,
                InputOption::VALUE_NONE,
                'if you want to create job'
            ),
        ];
        $this->setName('drop:pvs:export:rma');
        $this->setDescription('Manual rma export to Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $rmaId = $input->getOption(self::RMA_ENTITY_ID);
        $saveLocal = $input->getOption(self::SAVE_LOCAL) ?? false;
        $saveToFtp = $input->getOption(self::SAVE_TO_FTP) ?? false;
        $forced = $input->getOption(self::FORCED) ?? false;
        $createJob = $input->getOption(self::CREATE_JOB) ?? false;

        if (!$rma = $this->rmaRepository->get($rmaId)) {
            $output->writeln('<error>No rma was found.</error>');
            return;
        }

        $data = $this->jobExportRma->getPublisherData($rma, $saveLocal, $saveToFtp, $forced);

        if (!$createJob) {
            $this->exportRmaConsumer->process($data);
            return;
        }

        $this->publisher->publish(ExportRma::TOPIC_NAME, $data);
    }
}
