<?php


namespace Drop\Pvs\Model\Consumer\Export;

use Drop\Pvs\Api\Export\RmaInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Uploader;
use Drop\Pvs\Model\Rma\Authorize;
use Drop\Pvs\Model\Service\PdfDownloader;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Type;
use Magento\Customer\Model\Vat;
use Magento\Directory\Model\RegionFactory;
use Magento\Rma\Api\RmaRepositoryInterface;
use Magento\Rma\Model\Rma\Status\History;

class Rma
{
    private $configuration;
    private $utilities;
    private $rmaRepository;
    private $logger;
    private $vat;
    private $regionFactory;
    private $uploader;
    private $pdfDownloader;
    private $statusHistory;
    private $authorize;

    /**
     * Rma constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param RmaRepositoryInterface $rmaRepository
     * @param Logger $logger
     * @param Vat $vat
     * @param RegionFactory $regionFactory
     * @param Uploader $uploader
     * @param PdfDownloader $pdfDownloader
     * @param History $statusHistory
     * @param Authorize $authorize
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        RmaRepositoryInterface $rmaRepository,
        Logger $logger,
        Vat $vat,
        RegionFactory $regionFactory,
        Uploader $uploader,
        PdfDownloader $pdfDownloader,
        History $statusHistory,
        Authorize $authorize
    )
    {
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->rmaRepository = $rmaRepository;
        $this->logger = $logger;
        $this->vat = $vat;
        $this->regionFactory = $regionFactory;
        $this->uploader = $uploader;
        $this->pdfDownloader = $pdfDownloader;
        $this->statusHistory = $statusHistory;
        $this->authorize = $authorize;
    }

    /**
     * @param RmaInterface $data
     * @return bool
     */
    public function process(RmaInterface $data): bool
    {
        if (!$rma = $this->rmaRepository->get($data->getRmaId())) {
            $this->logger->error("rma not found [ID: {$data->getRmaId()}]");
        }

        // ordine già esportato e senza la richiesta di un export forzato
        if ($rma->getDropPvsExported() && !$data->getIsForced()) {
            return true;
        }

        $authorization = $this->authorize->execute($rma);
        if ($this->configuration->isRmaAuthorizeEmailEnabled() && $authorization) {
            $this->statusHistory->setRmaEntityId($rma->getEntityId());
            $this->statusHistory->sendAuthorizeEmail();
        }

        $this->logger->info("Exporting Rma [ID: {$rma->getEntityId()}]");

        // genero XDOCE
        $this->logger->info("Generate XDOCE [ID: {$rma->getEntityId()}]");
        $rmaData = $this->utilities->getXmlContentFromArray($this->generateRmaData($rma), "doce");

        $this->saveFiles($rma, $rmaData, $data->getSaveToFtp(), $data->getSaveLocal());

        $rma->setDropPvsExported(1);
        $this->rmaRepository->save($rma);

        return true;
    }

    /**
     * @param $rma
     * @return array
     */
    public function generateRmaData($rma): array
    {
        try {
            $rmaData = $this->getRmaFileItems($rma);
        } catch (\Exception $e) {
            $this->statusHistory->setRmaEntityId($rma->getId());
            $this->statusHistory->saveComment($e->getMessage(), false, true);
            $this->logger->error($e->getMessage());
        }

        return $rmaData;
    }

    /**
     * @param $rma
     * @return array
     */
    public function getRmaFileItems($rma): array
    {
        $items = [];
        foreach ($rma->getItems() as $item) {
            $items[] = [
                'riga' => [
                    'COD_SOCL'    => $this->configuration->getCodSocl(),
                    'COD_DICL'    => $this->configuration->getCodDicl(),
                    'COD_ART'     => $item->getProductSku(),
                    'QTA_RIGA'    => $item->getQtyRequested(),
                    'COD_UM'      => "PZ",
                    'ORDINE'      => $rma->getOrder()->getIncrementId() . '-0',
                    'COD_CLI_FOR' => $this->configuration->getCodCliFor(),
                    'BOLLA'       => $rma->getIncrementId(),
                    'DATA_BOLLA'  => $rma->getDateRequested(),
                    'COD_MCON'    => "",
                    'COD_CMOV'    => "RESO",
                ],
            ];
        }

        return $items;
    }

    /**
     * @param $rma
     * @param $rmaData
     * @param $saveToFtp
     * @param $saveLocal
     */
    public function saveFiles($rma, $rmaData, $saveToFtp, $saveLocal)
    {
        $saveOn = $saveToFtp && $saveLocal ? "Pvs FTP and Local" : ($saveToFtp ? "Pvs FTP" : "Local");
        try {
            // XDOCE
            $filename = $this->utilities->completeFilenameWithDateTime($this->configuration->getRmaFilename());
            $this->uploader->execute($filename, $rmaData, $saveToFtp, $saveLocal, "xml");
            $this->statusHistory->setRmaEntityId($rma->getId());
            $this->statusHistory->saveComment("XDOCE File [{$filename}] saved on {$saveOn}", false, true);
            $this->logger->info("XDOCE File [{$filename}] saved");
        } catch (\Exception $e) {
            $this->statusHistory->setRmaEntityId($rma->getId());
            $this->statusHistory->saveComment($e->getMessage(), false, true);
            $this->logger->error($e->getMessage());
        }
    }
}
