<?php

namespace Drop\Pvs\Model\Service;

use Drop\Pvs\Helper\ConfigurationHelper;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Serialize\SerializerInterface;
use Drop\Pvs\Logger\Logger;
use Magento\Sales\Api\OrderRepositoryInterface;

class PdfDownloader
{
    private $configuration;
    private $curl;
    private $serializer;
    private $logger;
    private $orderRepository;

    /**
     * PdfDownloader constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param Curl $curl
     * @param SerializerInterface $serializer
     * @param Logger $logger
     * @param OrderRepositoryInterface $orderRepository
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        Curl $curl,
        SerializerInterface $serializer,
        Logger $logger,
        OrderRepositoryInterface $orderRepository
    )
    {
        $this->configuration = $configurationHelper;
        $this->curl = $curl;
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->orderRepository = $orderRepository;
    }

    /**
     * @param $order
     * @return false|mixed
     */
    public function execute($order)
    {
        if (!$token = $this->getToken()) {
            $order->addStatusToHistory($order->getStatus(), "ERROR: can't retrieve oms token");
            $this->orderRepository->save($order);
            $this->logger->error("Can't retrieve oms token");
            return false;
        }

        $headers = ["Content-Type" => "application/json", "Authorization" => "Bearer " . $token];
        try {
            $this->curl->setHeaders($headers);
            $this->curl->get($this->configuration->getPdfInvoiceDownloadEndpoint() . $order->getIncrementId());
            $response = $this->serializer->unserialize($this->curl->getBody());

            if (!$response || !isset($response["success"])) {
                $order->addStatusToHistory($order->getStatus(), "ERROR - Cannot get PDF: {$this->curl->getBody()}");
                $this->orderRepository->save($order);
                $this->logger->error("Cannot get PDF: {$this->curl->getBody()}");
                return false;
            }

            return $response["data"];
        } catch (\Exception $e) {
            $order->addStatusToHistory($order->getStatus(), "EXCEPTION CURL: {$e->getMessage()}");
            $this->orderRepository->save($order);
            $this->logger->error("EXCEPTION CURL for order {$order->getIncrementId()}: {$e->getMessage()}");
            return false;
        }
    }

    /**
     * @return false|mixed
     */
    private function getToken()
    {
        $params = [
            'email'    => $this->configuration->getPdfInvoiceUsername(),
            'password' => $this->configuration->getPdfInvoicePassword()
        ];
        try {
            $this->curl->post($this->configuration->getPdfInoviceLoginEndpoint(), $params);

            $response = $this->serializer->unserialize($this->curl->getBody());
            if (!$response || !isset($response['success'])) {
                $this->logger->info("Cannot get token error was " . $this->curl->getBody());
                return false;
            }

            return $response["data"]["token"];
        } catch (\Exception $e) {
            $this->logger->info('CURL FAILED ' . $e->getMessage());
            return false;
        }
    }
}
