<?php


namespace Drop\Pvs\Console\Command\Export;


use Drop\Pvs\Model\Consumer\Export\Order as Consumer;
use Drop\Pvs\Model\Ftp\Uploader;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\MessageQueue\PublisherInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class InvoicePdf extends Command
{
    const ORDER_ENTITY_ID = 'entity-id';
    const SAVE_LOCAL = 'local';
    const SAVE_TO_FTP = 'ftp';
    const FORCED = 'forced';

    private $appState;
    private $orderRepository;
    private $consumer;
    private $uploader;

    /**
     * InvoicePdf constructor.
     * @param State $appState
     * @param OrderRepositoryInterface $orderRepository
     * @param Consumer $consumer
     * @param Uploader $uploader
     */
    public function __construct(
        State $appState,
        OrderRepositoryInterface $orderRepository,
        Consumer $consumer,
        Uploader $uploader
    )
    {
        $this->appState = $appState;
        $this->orderRepository = $orderRepository;
        $this->consumer = $consumer;
        $this->uploader = $uploader;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::ORDER_ENTITY_ID,
                null,
                InputOption::VALUE_REQUIRED,
                'get order entity id'
            ),
            new InputOption(
                self::SAVE_LOCAL,
                null,
                InputOption::VALUE_NONE,
                'if you want to save in local folder'
            ),
            new InputOption(
                self::SAVE_TO_FTP,
                null,
                InputOption::VALUE_NONE,
                'if you want to save on remote ftp folder'
            ),
            new InputOption(
                self::FORCED,
                null,
                InputOption::VALUE_NONE,
                'if you want to force export'
            ),
        ];
        $this->setName('drop:pvs:export:invoice-pdf');
        $this->setDescription('Manual order invoice pdf export to Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $orderId = $input->getOption(self::ORDER_ENTITY_ID);
        $saveLocal = $input->getOption(self::SAVE_LOCAL) ?? false;
        $saveToFtp = $input->getOption(self::SAVE_TO_FTP) ?? false;
        $forced = $input->getOption(self::FORCED) ?? false;

        if (!$order = $this->orderRepository->get($orderId)) {
            $output->writeln('<error>No order was found.</error>');
            return;
        }

        try {
            $pdfContent = $this->consumer->getPdfContent($order);
            if (!$pdfContent && !$forced) {
                $output->writeln('<warning>No pdf export needed for this order.</warning>');
                return;
            }

            if (isset($pdfContent['content'])) {
                $this->uploader->execute("{$order->getIncrementId()}-0", $pdfContent, $saveToFtp, $saveLocal, "pdf");
            } else {
                foreach ($pdfContent as $file) {
                    $this->uploader->execute($file['name'], $file, $saveToFtp, $saveLocal, "pdf");
                }
            }

            $saveOn = $saveToFtp && $saveLocal ? "Pvs FTP and Local" : ($saveToFtp ? "Pvs FTP" : "Local");
            $order->addStatusToHistory($order->getStatus(), "PDF Files manually saved on {$saveOn}");
            $this->orderRepository->save($order);
        } catch (\Exception $e) {
            $output->writeln("<error>{$e->getMessage()}</error>");
            return;
        }

        $output->writeln('<info>Export complete</info>');
    }
}
