<?php


namespace Drop\Pvs\Model\Consumer\Export;

use Stichoza\GoogleTranslate\GoogleTranslate;
use Drop\Pvs\Api\Export\OrderInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Uploader;
use Drop\Pvs\Model\Service\PdfDownloader;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Type;
use Magento\Customer\Model\Vat;
use Magento\Directory\Model\RegionFactory;
use Magento\Sales\Api\OrderRepositoryInterface;

class Order
{
    private $configuration;
    private $utilities;
    private $orderRepository;
    private $logger;
    private $vat;
    private $regionFactory;
    private $uploader;
    private $pdfDownloader;

    /**
     * Order constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param OrderRepositoryInterface $orderRepository
     * @param Logger $logger
     * @param Vat $vat
     * @param RegionFactory $regionFactory
     * @param Uploader $uploader
     * @param PdfDownloader $pdfDownloader
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        OrderRepositoryInterface $orderRepository,
        Logger $logger,
        Vat $vat,
        RegionFactory $regionFactory,
        Uploader $uploader,
        PdfDownloader $pdfDownloader
    )
    {
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->orderRepository = $orderRepository;
        $this->logger = $logger;
        $this->vat = $vat;
        $this->regionFactory = $regionFactory;
        $this->uploader = $uploader;
        $this->pdfDownloader = $pdfDownloader;
    }

    /**
     * @param OrderInterface $data
     * @return bool
     */
    public function process(OrderInterface $data): bool
    {
        if (!$order = $this->orderRepository->get($data->getOrderId())) {
            $this->logger->error("order not found [ID: {$data->getOrderId()}]");
        }

        // ordine già esportato e senza la richiesta di un export forzato
        if ($order->getDropPvsExported() && !$data->getIsForced()) {
            return true;
        }

        $this->logger->info("Exporting Order [ID: {$order->getEntityId()} - increment {$order->getIncrementId()}]");

        // recupero PDF
        $pdfContent = $this->getPdfContent($order);

        // genero XORDC
        $this->logger->info("Generate XORDC [ID: {$order->getEntityId()} - increment {$order->getIncrementId()}]");
        $orderData = $this->utilities->getXmlContentFromArray($this->generateOrderData($order), "orders");

        $this->saveFiles($order, $orderData, $pdfContent, $data->getSaveToFtp(), $data->getSaveLocal());

        $order->setDropPvsExported(1);
        $this->orderRepository->save($order);

        return true;
    }

    /**
     * @param $order
     * @return false|mixed
     */
    public function getPdfContent($order)
    {
        if (!$this->configuration->isPdfEnabled()) {
            return false;
        }

        $this->logger->info("[INVOICE PDF] Get Pdf [ID: {$order->getEntityId()} - increment {$order->getIncrementId()}]");
        return !$this->vat->isCountryInEU($order->getShippingAddress()->getCountryId()) ?
            $this->pdfDownloader->getInvoice($order) : false;
    }

    /**
     * @param $order
     * @return array
     */
    public function generateOrderData($order): array
    {
        try {
            $orderData = [
                'order' => [
                    'XORDC_TES'     => $this->getXordcHead($order),
                    'XORDC_CLIENTI' => $this->getXordcCustomer($order),
                    'righe'         => $this->getXordcRows($order),
                ]
            ];
        } catch (\Exception $e) {
            $order->addStatusToHistory($order->getStatus(), $e->getMessage());
            $this->orderRepository->save($order);
            $this->logger->error($e->getMessage());
        }

        return $orderData;
    }

    /**
     * @param $order
     * @return array
     */
    public function getXordcHead($order): array
    {
        $cod = 0;
        if (
            $this->configuration->isOrderPaymentCodEnabled() &&
            $this->configuration->getOrderPaymentCodCode() == $order->getPayment()->getMethodInstance()->getCode()
        ) {
            $cod = $order->getGrandTotal();
        }

        $itemsAmount = 0;
        foreach ($order->getAllVisibleItems() as $item) {
            $itemsAmount += (float)$item->getRowTotalInclTax();
        }
        $itemsAmount -= (float)$order->getBaseDiscountAmount();

        $insuredLimit = $this->configuration->getOrderInsuranceLimitValue($order->getStoreId());
        $insurance = ($insuredLimit && trim($insuredLimit) != '' && (float)$insuredLimit > 0 && $itemsAmount && (float)$itemsAmount > $insuredLimit) ? $itemsAmount : "";

        return [
            'COD_SOCL'        => $this->configuration->getCodSocl(),
            'COD_DICL'        => $this->configuration->getCodDicl(),
            'COD_ORDC'        => "{$order->getIncrementId()}-0",
            'COD_CLIENTE'     => "{$order->getIncrementId()}-0",
            'DATA_ORDC'       => $order->getCreatedAt(),
            'IMPORTO_COD'     => $cod,
            'IMPORTO_ASSIC'   => $insurance,
            'IMPORTO_FATTURA' => $order->getGrandTotal(),
            'VALUTA_FATTURA'  => $order->getOrderCurrencyCode(),
        ];
    }

    /**
     * @param $order
     * @return array
     */
    public function getXordcCustomer($order): array
    {
        $billing = $order->getBillingAddress();
        $shipping = $order->getBillingAddress();

        $vatId = ($billing->getRequestInvoice()) ? $billing->getVatId() : '';
        $street = (is_array($billing->getStreet())) ? implode(' ', $billing->getStreet()) : $billing->getStreet();
        $businessName = ($billing->getCompany()) ? $billing->getCompany() : $billing->getFirstname() . ' ' . $billing->getLastname();
        $city = $shipping->getCity();

        if (in_array($shipping->getCountryId(), $this->configuration->getOrderTranslationCountries())) {
            $tr = new GoogleTranslate();
            $businessName = $tr->translate($businessName);
            $street = $tr->translate($street);
            $city = $tr->translate($city);
        }

        return [
            'COD_SOCL'    => $this->configuration->getCodSocl(),
            'COD_DICL'    => $this->configuration->getCodDicl(),
            'COD_ORDC'    => "{$order->getIncrementId()}-0",
            'COD_CLIENTE' => "{$order->getIncrementId()}-0",
            'RAG_SOC'     => "<![CDATA[{$businessName}]]>",
            'EMAIL'       => $order->getCustomerEmail(),
            'INDIRIZZO'   => "<![CDATA[{$street}]]>",
            'INDIRIZZO2'  => '',
            'CAP'         => $shipping->getPostcode(),
            'CITTA'       => $city,
            'COD_PROV'    => substr($this->regionFactory->create()->load($shipping->getRegionId())->getCode(), 0, 2),
            'COD_NAZ'     => $shipping->getCountryId(),
            'PART_IVA'    => $vatId,
            'TELEFONO'    => $shipping->getTelephone(),
            'CELLULARE'   => '',
        ];
    }

    /**
     * @param $order
     * @return array
     */
    public function getXordcRows($order): array
    {
        $rows = [];
        $count = 1;
        foreach ($order->getAllVisibleItems() as $orderItem) {
            $rows[] = [
                'XORDC_RIGHE' => [
                    'COD_SOCL'         => $this->configuration->getCodSocl(),
                    'COD_DICL'         => $this->configuration->getCodDicl(),
                    'COD_ORDC'         => "{$order->getIncrementId()}-0",
                    'COD_ART'          => $orderItem->getSku(),
                    'QTA_RIGA'         => $orderItem->getQtyInvoiced(),
                    'RIGA_ORDC'        => $count,
                    'COD_MCON'         => '',
                    'SERVIZI_CORRIERE' => '',
                ],
            ];
            $count++;
        }

        return $rows;
    }

    /**
     * @param $order
     * @param $orderData
     * @param $pdfContent
     * @param $saveToFtp
     * @param $saveLocal
     */
    public function saveFiles($order, $orderData, $pdfContent, $saveToFtp, $saveLocal)
    {
        $saveOn = $saveToFtp && $saveLocal ? "Pvs FTP and Local" : ($saveToFtp ? "Pvs FTP" : "Local");
        try {
            // XORDC
            $filename = $this->utilities->completeFilenameWithDateTime($this->configuration->getOrderFilename());
            $this->uploader->execute($filename, $orderData, $saveToFtp, $saveLocal, "xml");
            $order->addStatusToHistory($order->getStatus(), "XORDC File [{$filename}] saved on {$saveOn}");
            $this->orderRepository->save($order);
            $this->logger->info("XORDC File [{$filename}] saved");

            if (!$pdfContent) {
                return;
            }

            if (isset($pdfContent['content'])) {
                $this->uploader->execute("{$order->getIncrementId()}-0", $pdfContent, $saveToFtp, $saveLocal, "pdf");
            } else {
                foreach ($pdfContent as $file) {
                    $this->uploader->execute($file['name'], $file, $saveToFtp, $saveLocal, "pdf");
                }
            }
            $order->addStatusToHistory($order->getStatus(), "PDF Files saved on {$saveOn}");
            $this->logger->info("PDF Files saved");
        } catch (\Exception $e) {
            $order->addStatusToHistory($order->getStatus(), $e->getMessage());
            $this->logger->error($e->getMessage());
        }
        $this->orderRepository->save($order);
    }
}
