<?php


namespace Drop\Pvs\Model\Consumer\Import;


use Drop\Pvs\Api\FileInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Modifier;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\Io\File;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\ShipmentRepositoryInterface;
use Magento\Sales\Model\Convert\Order;
use Magento\Sales\Model\Order\Email\Sender\ShipmentCommentSender;
use Magento\Sales\Model\Order\Shipment\Track;

class Shipment
{
    private $logger;
    private $configuration;
    private $utilities;
    private $modifier;
    private $csv;
    private $searchCriteriaBuilder;
    private $orderRepository;
    private $convertOrder;
    private $shipmentRepository;
    private $track;
    private $emailSender;

    /**
     * Shipment constructor.
     * @param Logger $logger
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param Modifier $modifier
     * @param Csv $csv
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param OrderRepositoryInterface $orderRepository
     * @param Order $convertOrder
     * @param ShipmentRepositoryInterface $shipmentRepository
     * @param Track $track
     * @param ShipmentCommentSender $emailSender
     */
    public function __construct(
        Logger $logger,
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        Modifier $modifier,
        Csv $csv,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        OrderRepositoryInterface $orderRepository,
        Order $convertOrder,
        ShipmentRepositoryInterface $shipmentRepository,
        Track $track,
        ShipmentCommentSender $emailSender
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->modifier = $modifier;
        $this->csv = $csv;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderRepository = $orderRepository;
        $this->convertOrder = $convertOrder;
        $this->shipmentRepository = $shipmentRepository;
        $this->track = $track;
        $this->emailSender = $emailSender;
    }

    /**
     * @param FileInterface $data
     * @throws \Exception
     */
    public function process(FileInterface $data)
    {
        if (!count($data->getFiles())) {
            return;
        }

        $carriers = $this->utilities->getCountriesCarrierArray();

        foreach ($data->getFiles() as $shipmentFile) {
            if (trim($this->configuration->getFtpRootPath() != "")) {
                $shipmentFile = str_replace($this->configuration->getFtpRootPath() . "/", "", $shipmentFile);
            }
            $expl = explode("/", $shipmentFile);
            $filename = array_pop($expl);
            $csvData = $this->utilities->getAssocArrayFromFile(implode("/", $expl), $filename, "\n", ",");

            foreach ($csvData as $row) {
                if (!isset($row['order_number'])) {
                    continue;
                }
                try {
                    $incrementId = str_replace("-0", "", $row['order_number']);
                    if (trim($incrementId) == "") {
                        continue;
                    }

                    $this->searchCriteriaBuilder->addFilter("increment_id", $incrementId);
                    foreach ($this->orderRepository->getList($this->searchCriteriaBuilder->create())->getItems() as $order) {
                        if (!$order->getCanShip()) {
                            continue;
                        }

                        $shipment = $this->convertOrder->toShipment($order);
                        foreach ($order->getAllItems() as $orderItem) {
                            $qtyShipped = $orderItem->getQtyOrdered();
                            $shipmentItem = $this->convertOrder->itemToShipmentItem($orderItem)->setQty($qtyShipped);
                            $shipment->addItem($shipmentItem);
                        }
                        $shipment->register();
                        $carrier = $carriers[$order->getShippingAddress()->getCountryid()] ?? $carriers['*'];
                        $this->track->addData([
                            'carrier_code' => $carrier,
                            'title'        => $carrier,
                            'number'       => $row['tracking_number'],
                            'description'  => $row['tracking_url']
                        ]);
                        $shipment->addTrack($this->track);
                        $this->shipmentRepository->save($shipment);
                        $order->setStatus($this->configuration->getOrderShippedStatus());
                        $order->addStatusToHistory($order->getStatus(), "Created shipment");
                        $this->orderRepository->save($order);
                        $this->emailSender->send($shipment);
                    }
                } catch (\Exception $exception) {
                    $this->logger->error($exception->getMessage());
                }
            }

//            $this->manageFtpFile($shipmentFile, $data->getCanDeleteFiles());
        }
    }

    /**
     * @param $filename
     * @param $canDeleteFile
     */
    protected function manageFtpFile($filename, $canDeleteFile)
    {
        if ($canDeleteFile) {
            $this->logger->info("delete file - {$filename}");
            $this->modifier->delete($filename);
            return;
        }

        $this->logger->info("move file to DONE - {$filename}");
        $this->modifier->moveToDone($filename);
    }
}
