<?php

namespace Drop\Pvs\Model\Consumer\Import;

use Drop\Pvs\Api\FileInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Downloader;
use Drop\Pvs\Model\Ftp\Modifier;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\Io\File;
use Magento\Sales\Api\OrderRepositoryInterface;

class Delivery
{
    const ACCEPTED_DELIVERED_STATUSES = [
        "CONSEGNATO",
        // TODO
    ];

    private $logger;
    private $configuration;
    private $utilities;
    private $modifier;
    private $searchCriteriaBuilder;
    private $orderRepository;

    /**
     * Delivery constructor.
     * @param Logger $logger
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param Modifier $modifier
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param OrderRepositoryInterface $orderRepository
     */
    public function __construct(
        Logger $logger,
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        Modifier $modifier,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        OrderRepositoryInterface $orderRepository
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->modifier = $modifier;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderRepository = $orderRepository;
    }

    /**
     * @param FileInterface $data
     * @throws \Exception
     */
    public function process(FileInterface $data)
    {
        if (!count($data->getFiles())) {
            return;
        }

        foreach ($data->getFiles() as $deliveryFile) {
            if (trim($this->configuration->getFtpRootPath() != "")) {
                $deliveryFile = str_replace($this->configuration->getFtpRootPath() . "/", "", $deliveryFile);
            }
            $expl = explode("/", $deliveryFile);
            $filename = array_pop($expl);
            $csvData = $this->utilities->getAssocArrayFromFile(implode("/", $expl), $filename, "\n");
            foreach ($csvData as $pvsOrder) {
                if (
                    !isset($pvsOrder['LAST_STATUS']) ||
                    !in_array($pvsOrder['LAST_STATUS'], self::ACCEPTED_DELIVERED_STATUSES)
                ) {
                    continue;
                }

                $incrementId = str_replace("-0", "", $pvsOrder['COD_ORDC']);
                if (trim($incrementId) == "") {
                    continue;
                }
                $this->searchCriteriaBuilder->addFilter("increment_id", $incrementId);
                foreach ($this->orderRepository->getList($this->searchCriteriaBuilder->create())->getItems() as $order) {
                    try {
                        $order->setStatus($this->configuration->getOrderDeliveredStatus());
                        $order->setDropPvsDeliveryDate(new \DateTime($pvsOrder["DTH_LAST_STATUS"]));
                        $order->addStatusToHistory($order->getStatus(), "Update delivery date");
                        $this->logger->info("Order #{$order->getIncrementId()} - Update delivery date on {$pvsOrder['DTH_LAST_STATUS']}");
                    } catch (\Exception $e) {
                        $this->logger->error($e->getMessage());
                        $order->addStatusToHistory($order->getStatus(), $e->getMessage());
                    }

                    $this->orderRepository->save($order);
                }
            }

            $this->manageFtpFile($deliveryFile, $data->getCanDeleteFiles());
        }
    }

    /**
     * @param $filename
     * @param $canDeleteFile
     */
    protected function manageFtpFile($filename, $canDeleteFile)
    {
        if ($canDeleteFile) {
            $this->logger->info("delete file - {$filename}");
            $this->modifier->delete($filename);
            return;
        }

        $this->logger->info("move file to DONE - {$filename}");
        $this->modifier->moveToDone($filename);
    }
}
