<?php


namespace Drop\Pvs\Helper;


use Drop\Pvs\Model\Ftp\Downloader;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Stdlib\DateTime\DateTimeFactory;
use Magento\Framework\Convert\ConvertArray;

class UtilitiesHelper extends AbstractHelper
{
    private $configuration;
    private $downloader;
    private $directoryList;
    private $driverFile;
    private $dateFactory;
    private $convertArray;

    public function __construct(
        ConfigurationHelper $configurationHelper,
        Downloader $downloader,
        Context $context,
        DirectoryList $directoryList,
        File $driverFile,
        DateTimeFactory $dateFactory,
        ConvertArray $convertArray
    )
    {
        $this->configuration = $configurationHelper;
        $this->downloader = $downloader;
        $this->directoryList = $directoryList;
        $this->driverFile = $driverFile;
        $this->dateFactory = $dateFactory;
        $this->convertArray = $convertArray;
        parent::__construct($context);
    }

    /**
     * @param $csvData
     * @return array
     */
    public function csvToArray($csvData): array
    {
        $fields = [];
        $result = [];
        $i = 0;
        foreach ($csvData as $row) {
            if (empty($fields)) {
                $fields = $row;
                continue;
            }
            foreach ($row as $k => $value) {
                $key = str_replace("\r", "", $fields[$k]);
                $key = str_replace("﻿", "", $key);
                $result[$i][$key] = str_replace("\r", "", htmlspecialchars($value));
            }
            $i++;
        }
        return $result;
    }

    /**
     * @param $dir
     * @return array
     */
    public function getLocalFiles($dir): array
    {
        $varPath = $this->directoryList->getPath(\Magento\Framework\App\Filesystem\DirectoryList::VAR_DIR);
        $pvsLocalFolder = "{$varPath}/pvs/tmp/{$dir}";

        return $this->driverFile->readDirectory($pvsLocalFolder);
    }

    /**
     * @param $xmlArray
     * @param $firstNode
     * @return bool|string
     */
    public function getXmlContentFromArray($xmlArray, $firstNode)
    {
        $content = new \SimpleXMLElement("<?xml version='1.0' encoding='UTF-8'?><{$firstNode}></{$firstNode}>");
        $this->array_to_xml($xmlArray, $content);

        return htmlspecialchars_decode($content->asXML());
    }

    /**
     * @param $data
     * @param $xml_data
     * @return mixed
     */
    public function array_to_xml($data, &$xml_data)
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                if (is_numeric($key)) {
                    $this->array_to_xml($value, $xml_data);
                    continue;
                }
                $subnode = $xml_data->addChild($key);
                $this->array_to_xml($value, $subnode);
            } else {
                $xml_data->addChild("$key", htmlspecialchars("$value"));
            }
        }

        return $xml_data;
    }

    /**
     * @param $filename
     * @return string
     */
    public function completeFilenameWithDateTime($filename): string
    {
        $datetime = $this->dateFactory->create()->gmtDate('YmdHis');
        $explName = explode('.', $filename);
        if (count($explName) == 1) {
            return "{$explName[0]}{$datetime}";
        }

        return "{$explName[0]}{$datetime}.{$explName[1]}";
    }

    /**
     * @param $content
     * @param string $columnDelimiter
     * @param string $rowDelimiter
     * @return array
     */
    public function convertCsvContentToArray($content, string $columnDelimiter = ";", string $rowDelimiter = "\n"): array
    {
        $data = [];
        $rows = explode($rowDelimiter, $content);
        foreach ($rows as $row) {
            $data[] = explode($columnDelimiter, $row);
        }

        return $data;
    }

    /**
     * @return array
     */
    public function getCountriesCarrierArray(): array
    {
        $result = [];
        $files = $this->downloader->list(
            $this->configuration->getCarrierFilename(),
            $this->configuration->getFtpCarrierDir()
        );

        foreach ($files as $filename) {

            $expl = explode("/", $filename);
            $content = $this->downloader->getFileContent($this->configuration->getFtpCarrierDir(), array_pop($expl));
            $contentArray = [];
            $rows = explode("\r\n", $content);
            foreach ($rows as $row) {
                $contentArray[] = explode(";", $row);
            }
            $data = $this->csvToArray($contentArray);
            foreach ($data as $row) {
                if (trim($row['Nazione']) == "") {
                    continue;
                }

                $result[$row['Nazione']] = $row['Vettore'];
            }
            break;
        }

        return $result;
    }

    /**
     * @param $ftpPath
     * @param $filename
     * @param $columnDelimiter
     * @param $rowDelimiter
     * @return array
     */
    public function getAssocArrayFromFile($ftpPath, $filename, $columnDelimiter = ";", $rowDelimiter = "\n"): array
    {
        $content = $this->downloader->getFileContent($ftpPath, $filename);

        $contentArray = $this->convertCsvContentToArray($content, $columnDelimiter, $rowDelimiter);

        return $this->csvToArray($contentArray);
    }

    /**
     * @param $ftpPath
     * @param $filename
     * @return mixed
     */
    public function getAssocArrayFromXml($ftpPath, $filename)
    {
        $fileData = $this->downloader->getFileContent($ftpPath, $filename);
        $xml = simplexml_load_string($fileData, "SimpleXMLElement", LIBXML_NOCDATA);
        $json = json_encode($xml);

        return json_decode($json, TRUE);
    }
}
