<?php


namespace Drop\Pvs\Model\Service;

use GuzzleHttp\Client;
use Magento\Framework\Filesystem;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Logger\Logger;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Rma\Model\Rma\Status\History;

class LdvDownloader
{
    const GSPED_DIR = "gsped";

    private $configuration;
    private $client;
    private $serializer;
    private $logger;
    private $filesystem;

    /**
     * LdvDownloader constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param Client $client
     * @param SerializerInterface $serializer
     * @param Logger $logger
     * @param Filesystem $filesystem
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        Client $client,
        SerializerInterface $serializer,
        Logger $logger,
        Filesystem $filesystem
    )
    {
        $this->configuration = $configurationHelper;
        $this->client = $client;
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->filesystem = $filesystem;
    }

    /**
     * @param $rma
     * @param $data
     * @return false|mixed
     */
    public function execute($rma, $data)
    {
        $order = $rma->getOrder();

        try {
            $response = $this->client->request(
                "POST",
                "{$this->configuration->getGspedApiUrl()}/shipment",
                [
                    "headers"         => [
                        "Content-Type" => "application/json",
                        "Accept"       => "application/json",
                        "X-API-KEY"    => $this->configuration->getGspedApiKey()
                    ],
                    "json"            => $data,
                    "connect_timeout" => $this->configuration->getGspedConnectionTimeout(),
                    "timeout"         => $this->configuration->getGspedTimeout(),
                    "read_timeout"    => $this->configuration->getGspedReadTimeout(),

                ]
            );

            if (!$response || $response->getStatusCode() != 200) {
                $this->logger->error("[PVS][GSPED-LDV] Cannot get LDV: {$this->curl->getBody()}");
                return false;
            }

            $response = json_decode($response->getBody());

            if ($this->configuration->isGspedSaveLabelEnabled()) {
                $this->saveLabel($response);
            }

            return $response->label_pdf[0];
        } catch (\Exception $e) {
            $this->logger->error("[PVS][GSPED-LDV] EXCEPTION CURL for rma LDV request {$rma->getIncrementId()}: {$e->getMessage()}");
            return false;
        }
    }

    /**
     * @param $body
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    protected function saveLabel($body)
    {
        $labels = $body->label_pdf;

        if (!count($labels)) {
            $msgError = "No labels for rem {$body->ddt_alpha}";
            $this->logger->error("No labels for rem {$body->ddt_alpha}");
            throw new \Exception($msgError);
        }

        foreach ($labels as $index => $label) {
            // Filename
            $fileName = date("Ymd_His") . "_" . $body->ddt_num . "_" . $index . ".pdf";

            // Decode label
            $decoded = base64_decode($label);

            // Set directory
            $writer = $this->filesystem->getDirectoryWrite(DirectoryList::PUB);

            // Create label file and write
            $file = $writer->openFile(self::GSPED_DIR . DIRECTORY_SEPARATOR . $fileName, 'w');
            try {
                $file->lock();
                try {
                    $file->write($decoded);
                } finally {
                    $file->unlock();
                }
            } finally {
                $file->close();
            }
        }
    }
}
