<?php


namespace Drop\Pvs\Model\Ftp;


use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Logger\Logger;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Io\Ftp;
use Magento\Framework\Filesystem\Io\File;

class Uploader
{
    private $logger;
    private $configuration;
    private $csvProcessor;
    private $ftp;
    private $io;
    private $directoryList;

    /**
     * Uploader constructor.
     * @param Logger $logger
     * @param ConfigurationHelper $configuration
     * @param Csv $csvProcessor
     * @param Ftp $ftp
     */
    public function __construct(
        Logger $logger,
        ConfigurationHelper $configuration,
        Csv $csvProcessor,
        Ftp $ftp,
        File $io,
        DirectoryList $directoryList
    )
    {
        $this->logger = $logger;
        $this->configuration = $configuration;
        $this->csvProcessor = $csvProcessor;
        $this->ftp = $ftp;
        $this->io = $io;
        $this->directoryList = $directoryList;
    }

    /**
     * @param string $filename
     * @param $content
     * @param bool $saveToFtp
     * @param bool $saveLocal
     * @param string $extension
     */
    public function execute(string $filename, $content, bool $saveToFtp = true, bool $saveLocal = false, string $extension = 'csv')
    {
        if (
            ((is_array($content) || is_object($content)) && !count($content)) ||
            (is_string($content) && trim($content) == "")
        ) {
            $this->logger->info("No content for " . $filename);

            return;
        }

        $filename = $filename . "." . $extension;
        $localTmpFile = $this->getLocalFile($filename);

        switch ($extension) {
            case 'csv':
                $this->addCsvHeader($content);
                $this->csvProcessor
                    ->setDelimiter($this->configuration->getCsvSeparator())
                    ->setEnclosure($this->configuration->getCsvEnclosure())
                    ->appendData($localTmpFile, $content);
                break;
            case 'pdf':
                $content = base64_decode($content["content"]);
                $this->io->write($localTmpFile, $content);
                break;
            case 'xml':
            default:
                $this->io->write($localTmpFile, $content);
                break;
        }

        if ($saveToFtp) {
            $ftpParams = [
                'host'     => $this->configuration->getFtpHost(),
                'port'     => $this->configuration->getFtpPort(),
                'user'     => $this->configuration->getFtpUsername(),
                'password' => $this->configuration->getFtpPassword(),
                'path'     => $this->configuration->getFtpRootPath(),
                'passive'  => true
            ];
            try {
                $this->ftp->open($ftpParams);
                $this->ftp->cd($this->configuration->getFtpWriteDir());
                $this->ftp->write($filename, $localTmpFile);
                $this->ftp->close();
                $this->logger->info("Uploaded file " . $filename);
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());
            }
        }

        if (!$saveLocal) {
            $this->io->rm($localTmpFile);
        }
    }

    /**
     * @param $filename
     * @return string
     * @throws \Magento\Framework\Exception\FileSystemException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function getLocalFile($filename): string
    {
        $varPath = $this->directoryList->getPath(\Magento\Framework\App\Filesystem\DirectoryList::VAR_DIR);
        $localFileFolder = "{$varPath}/pvs";
        $this->io->checkAndCreateFolder($localFileFolder);

        return "{$localFileFolder}/{$filename}";
    }

    /**
     * @param $content
     * @return int
     */
    public function addCsvHeader(&$content)
    {
        $header = [];
        foreach ($content as $row) {
            foreach ($row as $column => $value) {
                $header[] = $column;
            }
            return array_unshift($content, $header);
        }
    }
}
