<?php

namespace Drop\Pvs\Console\Command\Export;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\App\State;
use Magento\Framework\App\Area;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Framework\MessageQueue\PublisherInterface;
use \Drop\Pvs\Cron\Export\Order as exportOrder;
use \Drop\Pvs\Model\Consumer\Export\Order as ExportOrderConsumer;

class Order extends Command
{
    const ORDER_ENTITY_ID = 'entity-id';
    const SAVE_LOCAL = 'local';
    const SAVE_TO_FTP = 'ftp';
    const FORCED = 'forced';
    const CREATE_JOB = 'create-job';

    /**
     * @var State
     */
    private $appState;
    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;
    /**
     * @var PublisherInterface
     */
    private $publisher;
    /**
     * @var exportOrder
     */
    private $jobExportOrder;
    /**
     * @var ExportOrderConsumer
     */
    private $exportOrderConsumer;

    /**
     * Order constructor.
     * @param OrderRepositoryInterface $orderRepository
     * @param PublisherInterface $publisher
     * @param exportOrder $jobExportOrder
     * @param State $appState
     * @param ExportOrderConsumer $exportOrderConsumer
     */
    public function __construct(
        OrderRepositoryInterface $orderRepository,
        PublisherInterface $publisher,
        exportOrder $jobExportOrder,
        State $appState,
        ExportOrderConsumer $exportOrderConsumer
    )
    {
        $this->appState = $appState;
        $this->orderRepository = $orderRepository;
        $this->publisher = $publisher;
        $this->jobExportOrder = $jobExportOrder;
        $this->exportOrderConsumer = $exportOrderConsumer;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [
            new InputOption(
                self::ORDER_ENTITY_ID,
                null,
                InputOption::VALUE_REQUIRED,
                'get order entity id'
            ),
            new InputOption(
                self::SAVE_LOCAL,
                null,
                InputOption::VALUE_NONE,
                'if you want to save in local folder'
            ),
            new InputOption(
                self::SAVE_TO_FTP,
                null,
                InputOption::VALUE_NONE,
                'if you want to save on remote ftp folder'
            ),
            new InputOption(
                self::FORCED,
                null,
                InputOption::VALUE_NONE,
                'if you want to force export'
            ),
            new InputOption(
                self::CREATE_JOB,
                null,
                InputOption::VALUE_NONE,
                'if you want to create job'
            ),
        ];
        $this->setName('drop:pvs:export:order');
        $this->setDescription('Manual order export to Pvs [directly or via job]');
        $this->setDefinition($options);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->appState->setAreaCode(Area::AREA_ADMINHTML);

        $orderId = $input->getOption(self::ORDER_ENTITY_ID);
        $saveLocal = $input->getOption(self::SAVE_LOCAL) ?? false;
        $saveToFtp = $input->getOption(self::SAVE_TO_FTP) ?? false;
        $forced = $input->getOption(self::FORCED) ?? false;
        $createJob = $input->getOption(self::CREATE_JOB) ?? false;

        if (!$order = $this->orderRepository->get($orderId)) {
            $output->writeln('<error>No order was found.</error>');
            return;
        }
        
        $data = $this->jobExportOrder->getPublisherData($order, $saveLocal, $saveToFtp, $forced);

        if (!$createJob) {
            $this->exportOrderConsumer->process($data);
            return;
        }

        $this->publisher->publish(exportOrder::TOPIC_NAME, $data);
    }
}
