<?php

namespace Drop\Pvs\Model\Rma\Status;

use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Model\Mail\Template\TransportBuilder;
use Drop\Pvs\Model\Service\LdvDownloader;
use Drop\Pvs\Model\Service\PdfDownloader;
use Magento\Directory\Model\RegionFactory;
use Magento\Framework\Api\AttributeValueFactory;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Rma\Api\RmaAttributesManagementInterface;
use Magento\Rma\Api\RmaRepositoryInterface;
use Magento\Rma\Model\Rma;
use Magento\Rma\Model\Rma\Status\History as MagentoHistory;
use Magento\Sales\Model\Order\Address\Renderer as AddressRenderer;

class History extends MagentoHistory
{
    protected $configuration;
    protected $pdfDownloader;
    protected $ldvDownloader;
    protected $regionFactory;
    protected $transportBuilder;


    public function __construct(
        ConfigurationHelper $configurationHelper,
        PdfDownloader $pdfDownloader,
        LdvDownloader $ldvDownloader,
        RegionFactory $regionFactory,
        TransportBuilder $dropTransportBuilder,
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        AttributeValueFactory $customAttributeFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Rma\Model\RmaFactory $rmaFactory,
        \Magento\Rma\Model\Config $rmaConfig,
        \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder,
        \Magento\Framework\Stdlib\DateTime\DateTime $dateTimeDateTime,
        \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation,
        \Magento\Rma\Helper\Data $rmaHelper,
        TimezoneInterface $localeDate,
        RmaRepositoryInterface $rmaRepositoryInterface,
        RmaAttributesManagementInterface $metadataService,
        AddressRenderer $addressRenderer,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    )
    {
        $this->configuration = $configurationHelper;
        $this->pdfDownloader = $pdfDownloader;
        $this->ldvDownloader = $ldvDownloader;
        $this->regionFactory = $regionFactory;
        $this->transportBuilder = $dropTransportBuilder;

        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $storeManager,
            $rmaFactory,
            $rmaConfig,
            $transportBuilder,
            $dateTimeDateTime,
            $inlineTranslation,
            $rmaHelper,
            $localeDate,
            $rmaRepositoryInterface,
            $metadataService,
            $addressRenderer,
            $resource,
            $resourceCollection,
            $data
        );
    }

    /**
     * @param Rma $rma
     * @param string $rootConfig
     * @return $this|History
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function _sendRmaEmailWithItems(Rma $rma, $rootConfig)
    {
        $storeId = $rma->getStoreId();
        $order = $rma->getOrder();

        $this->_rmaConfig->init($rootConfig, $storeId);
        if (!$this->_rmaConfig->isEnabled()) {
            return $this;
        }

        $this->inlineTranslation->suspend();

        $copyTo = $this->_rmaConfig->getCopyTo();
        $copyMethod = $this->_rmaConfig->getCopyMethod();

        if ($order->getCustomerIsGuest()) {
            $template = $this->_rmaConfig->getGuestTemplate();
            $customerName = $order->getBillingAddress()->getName();
        } else {
            $template = $this->_rmaConfig->getTemplate();
            $customerName = $rma->getCustomerName();
        }

        $sendTo = [['email' => $order->getCustomerEmail(), 'name' => $customerName]];
        if ($rma->getCustomerCustomEmail() && $rma->getCustomerCustomEmail() !== $order->getCustomerEmail()) {
            $sendTo[] = ['email' => $rma->getCustomerCustomEmail(), 'name' => $customerName];
        }
        if ($copyTo && $copyMethod == 'copy') {
            foreach ($copyTo as $email) {
                $sendTo[] = ['email' => $email, 'name' => null];
            }
        }

        $returnAddress = $this->rmaHelper->getReturnAddress('html', [], $storeId);

        $bcc = [];
        if ($copyTo && $copyMethod == 'bcc') {
            $bcc = $copyTo;
        }
        $store = $this->_storeManager->getStore($storeId);
        $identity = $this->_rmaConfig->getIdentity('', $storeId);

        try {
            foreach ($sendTo as $recipient) {
                $this->transportBuilder->setTemplateIdentifier($template)
                    ->setTemplateOptions(['area' => \Magento\Framework\App\Area::AREA_FRONTEND, 'store' => $storeId])
                    ->setTemplateVars(
                        [
                            'rma'                      => $rma,
                            'rma_id'                   => $rma->getId(),
                            'rma_data'                 => [
                                'status_label' => is_string($rma->getStatusLabel()) ?
                                    $rma->getStatusLabel() : $rma->getStatusLabel()->render(),
                            ],
                            'order'                    => $order,
                            'order_data'               => [
                                'customer_name' => $order->getCustomerName(),
                            ],
                            'created_at_formatted_1'   => $rma->getCreatedAtFormated(1),
                            'store'                    => $store,
                            'return_address'           => $returnAddress,
                            'item_collection'          => $rma->getItemsForDisplay(),
                            'formattedShippingAddress' => $this->addressRenderer->format(
                                $order->getShippingAddress(),
                                'html'
                            ),
                            'formattedBillingAddress'  => $this->addressRenderer->format(
                                $order->getBillingAddress(),
                                'html'
                            ),
                            'supportEmail'             => $store->getConfig('trans_email/ident_support/email'),
                            'storePhone'               => $store->getConfig('general/store_information/phone'),
                        ]
                    )
                    ->setFromByScope($identity, $storeId)
                    ->addTo($recipient['email'], $recipient['name'])
                    ->addBcc($bcc)
                    ->setReplyTo($sendTo[0]['email']);

                if ($this->configuration->isPdfEnabled()) {
                    if ($pdfFile = $this->pdfDownloader->getRmaProforma($rma)) {
                        $this->transportBuilder->addAttachment(base64_decode($pdfFile['content']), $pdfFile['name']);
                    } else {
                        $this->saveComment("ERROR: retrieve rma proforma", false, true);
                    }
                }

                if ($this->configuration->isGspedEnabled()) {
                    if ($ldvFileContent = $this->ldvDownloader->execute($rma, $this->getLdvData($rma))) {
                        $this->transportBuilder->addAttachment(base64_decode($ldvFileContent), "ldv.pdf");
                    } else {
                        $this->saveComment("ERROR: retrieve ldv", false, true);
                    }
                }

                $this->transportBuilder->getTransport()->sendMessage();
            }

            $this->setEmailSent(true);
        } catch (MailException $exception) {
            $this->_logger->critical($exception->getMessage());
        }

        $this->inlineTranslation->resume();

        return $this;
    }

    /**
     * @param $rma
     * @return array
     */
    public function getLdvData($rma)
    {
        $order = $rma->getOrder();
        $address = $order->getShippingAddress();
        list($carrierCode, $carrierService) = $this->configuration->getGspedCountryRmaCarrierService($address->getCountryId());

        return [
            "client_id"           => $this->configuration->getGspedClientId(),
            "colli"               => 1,
            "corriere"            => $carrierCode,
            "servizio"            => $carrierService,
            "ddt_alpha"           => "CRN{$order->getIncrementId()}",
            "flag_reso"           => 1,
            "peso"                => 0.9,
            "date_req_ritiro"     => date("Y-m-d H:i:s", strtotime($rma->getDateRequested() . " +1 weekdays")),
            "opening_time"        => "08:00",
            "closing_time"        => "20:00",
            "rcpt_addr"           => "{$this->configuration->getMageRmaStreet()} {$this->configuration->getMageRmaStreet2()}",
            "rcpt_cap"            => $this->configuration->getMageRmaPostcode(),
            "rcpt_city"           => $this->configuration->getMageRmaCity(),
            "rcpt_contact"        => $this->configuration->getMageRmaName(),
            "rcpt_country_code"   => $this->configuration->getMageRmaCountryId(),
            "rcpt_email"          => $this->configuration->getMageRmaEmail(),
            "rcpt_name"           => $this->configuration->getMageRmaName(),
            "rcpt_phone"          => $this->configuration->getMageRmaPhone(),
            "rcpt_prov"           => $this->regionFactory->create()->load($this->configuration->getMageRmaRegionId())->getCode(),
            "sender_addr"         => is_array($address->getStreet()) ? implode(" ", $address->getStreet()) : $address->getStreet(),
            "sender_cap"          => $address->getPostcode(),
            "sender_city"         => $address->getCity(),
            "sender_contact"      => "{$address->getLastname()} {$address->getFirstname()}",
            "sender_country_code" => $address->getCountryId(),
            "sender_email"        => $address->getEmail(),
            "sender_name"         => "{$address->getLastname()} {$address->getFirstname()}",
            "sender_phone"        => $address->getTelephone(),
            "sender_prov"         => $this->regionFactory->create()->load($address->getRegionId())->getCode(),
        ];
    }
}
