<?php

namespace Drop\Pvs\Model\Rma\Status;

use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Model\Mail\Template\TransportBuilder;
use Drop\Pvs\Model\Service\LdvDownloader;
use Drop\Pvs\Model\Service\PdfDownloader;
use Drop\Pvs\Logger\Logger;
use Magento\Directory\Model\RegionFactory;
use Magento\Framework\Api\AttributeValueFactory;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Rma\Api\RmaAttributesManagementInterface;
use Magento\Rma\Api\RmaRepositoryInterface;
use Magento\Rma\Model\Rma;
use Magento\Rma\Model\Rma\Status\History as MagentoHistory;
use Magento\Sales\Model\Order\Address\Renderer as AddressRenderer;

class History extends MagentoHistory
{
    protected $configuration;
    protected $pdfDownloader;
    protected $ldvDownloader;
    protected $regionFactory;
    protected $transportBuilder;
    private $logger;


    public function __construct(
        Logger $logger,
        ConfigurationHelper $configurationHelper,
        PdfDownloader $pdfDownloader,
        LdvDownloader $ldvDownloader,
        RegionFactory $regionFactory,
        TransportBuilder $dropTransportBuilder,
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        AttributeValueFactory $customAttributeFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Rma\Model\RmaFactory $rmaFactory,
        \Magento\Rma\Model\Config $rmaConfig,
        \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder,
        \Magento\Framework\Stdlib\DateTime\DateTime $dateTimeDateTime,
        \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation,
        \Magento\Rma\Helper\Data $rmaHelper,
        TimezoneInterface $localeDate,
        RmaRepositoryInterface $rmaRepositoryInterface,
        RmaAttributesManagementInterface $metadataService,
        AddressRenderer $addressRenderer,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->pdfDownloader = $pdfDownloader;
        $this->ldvDownloader = $ldvDownloader;
        $this->regionFactory = $regionFactory;
        $this->transportBuilder = $dropTransportBuilder;

        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $storeManager,
            $rmaFactory,
            $rmaConfig,
            $transportBuilder,
            $dateTimeDateTime,
            $inlineTranslation,
            $rmaHelper,
            $localeDate,
            $rmaRepositoryInterface,
            $metadataService,
            $addressRenderer,
            $resource,
            $resourceCollection,
            $data
        );
    }

    /**
     * @param Rma $rma
     * @param string $rootConfig
     * @return $this|History
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function _sendRmaEmailWithItems(Rma $rma, $rootConfig)
    {
        $storeId = $rma->getStoreId();
        $order = $rma->getOrder();
        $this->logger->info("[EMAIL NOTIFY RETURN] START - RMA: " . $rma->getIncrementId() . " - ORDER: " . $order->getIncrementId());

        $this->_rmaConfig->init($rootConfig, $storeId);
        // se è disabilitata la mail
        // se non è uno stato in cui noi inviamo mail (nuovo o autorizzato o approvato)
        // se è stato processed_and_close (quindi approvato) e non è abilitata la mail da configurazione
        if (
            !$this->_rmaConfig->isEnabled() ||
            (
                $rma->getStatus() != Rma\Source\Status::STATE_PENDING &&
                $rma->getStatus() != Rma\Source\Status::STATE_AUTHORIZED &&
                $rma->getStatus() != Rma\Source\Status::STATE_PROCESSED_CLOSED
            ) ||
            (
                $rma->getStatus() == Rma\Source\Status::STATE_PROCESSED_CLOSED &&
                !$this->configuration->isReturnApprovedEmailEnabled()
            )
        ) {
            $this->logger->info("[EMAIL NOTIFY RETURN] STOP - requirements not met - RMA: " . $rma->getIncrementId() . " - RMA STATUS: " . $rma->getStatus() . " - ORDER: " . $order->getIncrementId());
            return $this;
        }

        $this->inlineTranslation->suspend();

        $copyTo = $this->_rmaConfig->getCopyTo();
        $copyMethod = $this->_rmaConfig->getCopyMethod();

        if ($order->getCustomerIsGuest()) {
            $template = $this->_rmaConfig->getGuestTemplate();
            $customerName = $order->getBillingAddress()->getName();
        } else {
            $template = $this->_rmaConfig->getTemplate();
            $customerName = $rma->getCustomerName();
        }
        // in caso di stato "processed and closed" metto il template custom sia per guest che non
        if ($rma->getStatus() === Rma\Source\Status::STATE_PROCESSED_CLOSED) {
            $template = $this->configuration->getReturnApprovedEmailTemplate($storeId);
        }

        $sendTo = [['email' => $order->getCustomerEmail(), 'name' => $customerName]];
        if ($rma->getCustomerCustomEmail() && $rma->getCustomerCustomEmail() !== $order->getCustomerEmail()) {
            $sendTo[] = ['email' => $rma->getCustomerCustomEmail(), 'name' => $customerName];
        }
        if ($copyTo && $copyMethod == 'copy') {
            foreach ($copyTo as $email) {
                $sendTo[] = ['email' => $email, 'name' => null];
            }
        }

        $returnAddress = $this->rmaHelper->getReturnAddress('html', [], $storeId);

        $bcc = [];
        if ($copyTo && $copyMethod == 'bcc') {
            $bcc = $copyTo;
        }
        $store = $this->_storeManager->getStore($storeId);
        $identity = $this->_rmaConfig->getIdentity('', $storeId);

        try {
            // se lo stato è quello selezionato per inviare gli allegati, allora li genero e allego
            $pdfFile = $ldvFileContent = null;
            if ($rma->getStatus() == $this->configuration->getReturnStatusEmailAttachments()) {
                if ($this->configuration->isPdfEnabled()) {
                    if (!$pdfFile = $this->pdfDownloader->getRmaProforma($rma)) {
                        $this->saveComment("ERROR: retrieve rma proforma", false, true);
                        $this->logger->error("[EMAIL NOTIFY RETURN] ERROR: retrieve rma proforma - RMA: " . $rma->getIncrementId());
                    }
                }

                if ($this->configuration->isGspedEnabled()) {
                    if (!$ldvFileContent = $this->ldvDownloader->execute($rma, $this->getLdvData($rma))) {
                        $this->saveComment("ERROR: retrieve ldv", false, true);
                        $this->logger->error("[EMAIL NOTIFY RETURN] ERROR: retrieve ldv - RMA: " . $rma->getIncrementId());
                    }
                }
            }
            foreach ($sendTo as $recipient) {
                $this->transportBuilder->setTemplateIdentifier($template)
                    ->setTemplateOptions(['area' => \Magento\Framework\App\Area::AREA_FRONTEND, 'store' => $storeId])
                    ->setTemplateVars(
                        [
                            'rma'                      => $rma,
                            'rma_id'                   => $rma->getId(),
                            'rma_data'                 => [
                                'status_label' => is_string($rma->getStatusLabel()) ?
                                    $rma->getStatusLabel() : $rma->getStatusLabel()->render(),
                            ],
                            'order'                    => $order,
                            'order_data'               => [
                                'customer_name' => $order->getCustomerName(),
                            ],
                            'created_at_formatted_1'   => $rma->getCreatedAtFormated(1),
                            'store'                    => $store,
                            'return_address'           => $returnAddress,
                            'item_collection'          => $rma->getItemsForDisplay(),
                            'formattedShippingAddress' => $this->addressRenderer->format(
                                $order->getShippingAddress(),
                                'html'
                            ),
                            'formattedBillingAddress'  => $this->addressRenderer->format(
                                $order->getBillingAddress(),
                                'html'
                            ),
                            'supportEmail'             => $store->getConfig('trans_email/ident_support/email'),
                            'storePhone'               => $store->getConfig('general/store_information/phone'),
                        ]
                    )
                    ->setFromByScope($identity, $storeId)
                    ->addTo($recipient['email'], $recipient['name'])
                    ->addBcc($bcc)
                    ->setReplyTo($sendTo[0]['email']);

                if ($pdfFile) {
                    $this->transportBuilder->addAttachment(base64_decode($pdfFile['content']), $pdfFile['name']);
                    $this->logger->info("[EMAIL NOTIFY RETURN] retrieve INVOICE: " . $pdfFile['name'] . " - RMA: " . $rma->getIncrementId());
                }
                if ($ldvFileContent) {
                    $this->transportBuilder->addAttachment(base64_decode($ldvFileContent), "ldv.pdf");
                    $this->logger->info("[EMAIL NOTIFY RETURN] retrieve LDV: ldv.pdf - RMA: " . $rma->getIncrementId());
                }

                try {
                    $this->transportBuilder->getTransport()->sendMessage();
                } catch (Exception $e) {
                    $this->logger->error("[EMAIL NOTIFY RETURN] RMA: {$rma->getIncrementId()} - ERROR: {$e->getMessage()}");
                }

                $this->logger->info("[EMAIL NOTIFY RETURN] SENT RMA email SCOPE: " . json_encode($identity) . " - TO: " . json_encode($recipient['email']) . " - BCC: " . json_encode($bcc) . " - REPLY-TO: " . json_encode($sendTo[0]['email']));
            }

            $this->setEmailSent(true);
        } catch (MailException $e) {
            $this->logger->error("[EMAIL NOTIFY RETURN] RMA: {$rma->getIncrementId()} - ERROR: {$e->getMessage()}");
        }

        $this->inlineTranslation->resume();

        $pdfFile = $ldvFileContent = null;

        $this->logger->info("[EMAIL NOTIFY RETURN] END - RMA: " . $rma->getIncrementId() . " - ORDER: " . $order->getIncrementId());

        return $this;
    }

    /**
     * @param $rma
     * @return array
     */
    public function getLdvData($rma)
    {
        $order = $rma->getOrder();
        $address = $order->getShippingAddress();
        list($carrierCode, $carrierService) = $this->configuration->getGspedCountryRmaCarrierService($address->getCountryId());

        return [
            "client_id"           => $this->configuration->getGspedClientId(),
            "colli"               => 1,
            "corriere"            => $carrierCode,
            "servizio"            => $carrierService,
            "ddt_alpha"           => "{$this->configuration->getGspedDdtPrefix()}{$order->getIncrementId()}",
            "flag_reso"           => 1,
            "peso"                => 0.9,
            "date_req_ritiro"     => date("Y-m-d H:i:s", strtotime($rma->getDateRequested() . " +1 weekdays")),
            "opening_time"        => "08:00:00",
            "closing_time"        => "20:00:00",
            "rcpt_addr"           => "{$this->configuration->getMageRmaStreet()} {$this->configuration->getMageRmaStreet2()}",
            "rcpt_cap"            => $this->configuration->getMageRmaPostcode(),
            "rcpt_city"           => $this->configuration->getMageRmaCity(),
            "rcpt_contact"        => $this->configuration->getMageRmaName(),
            "rcpt_country_code"   => $this->configuration->getMageRmaCountryId(),
            "rcpt_email"          => $this->configuration->getMageRmaEmail(),
            "rcpt_name"           => $this->configuration->getMageRmaName(),
            "rcpt_phone"          => $this->configuration->getMageRmaPhone(),
            "rcpt_prov"           => $this->regionFactory->create()->load($this->configuration->getMageRmaRegionId())->getCode(),
            "sender_addr"         => is_array($address->getStreet()) ? implode(" ", $address->getStreet()) : $address->getStreet(),
            "sender_cap"          => $address->getPostcode(),
            "sender_city"         => $address->getCity(),
            "sender_contact"      => "{$address->getLastname()} {$address->getFirstname()}",
            "sender_country_code" => $address->getCountryId(),
            "sender_email"        => $address->getEmail(),
            "sender_name"         => substr("{$address->getLastname()} {$address->getFirstname()}", 0, 35),
            "sender_phone"        => $address->getTelephone(),
            "sender_prov"         => substr($this->regionFactory->create()->load($address->getRegionId())->getCode(), 0, 2),
            "merce"               => "clothes",
        ];
    }
}
