<?php


namespace Drop\Pvs\Cron\Export;

use Drop\Pvs\Api\Export\OrderInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Consumer\Export\Order as Consumer;
use Drop\Pvs\Model\Publisher;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Bulk\BulkManagementInterface;
use Magento\Framework\Bulk\OperationInterface;
use Magento\Framework\DataObject\IdentityGeneratorInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Framework\MessageQueue\PublisherInterface;

class Order
{
    const TOPIC_NAME = 'pvs.export.order';

    private $publisher;
    private $consumer;
    private $searchCriteriaBuilder;
    private $configuration;
    private $utilities;
    private $orderRepository;
    private $orderInterface;

    /**
     * Order constructor.
     * @param Publisher $publisher
     * @param Consumer $consumer
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param OrderRepositoryInterface $orderRepository
     * @param OrderInterface $orderInterface
     */
    public function __construct(
        Publisher $publisher,
        Consumer $consumer,
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        OrderRepositoryInterface $orderRepository,
        OrderInterface $orderInterface
    )
    {
        $this->publisher = $publisher;
        $this->consumer = $consumer;
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderRepository = $orderRepository;
        $this->orderInterface = $orderInterface;
    }

    public function execute()
    {
        if (!$this->configuration->isEnabled() || !$this->configuration->isOrderExportEnabled() || !$this->configuration->isCommunicationEnabled()) {
            return;
        }

        // recupero tutti gli ordini non esportati con lo stato definito da bo
        $this->searchCriteriaBuilder->addFilter("status", $this->configuration->getOrderStatusToExport());
        $this->searchCriteriaBuilder->addFilter("drop_pvs_exported", 0);
        if (!$orders = $this->orderRepository->getList($this->searchCriteriaBuilder->create())->getItems()) {
            return;
        }

        foreach ($orders as $order) {
            sleep(1);
            if ($this->utilities->checkIsGiftCard($order)) {
                $order->setDropPvsExported(1);
                $this->orderRepository->save($order);
                continue;
            }


            $publisherData = $this->getPublisherData($order);

            if (!$this->configuration->isJobEnabled()) {
                $this->consumer->process($publisherData);
                continue;
            }

            $this->publisher->publish(self::TOPIC_NAME, $publisherData, "Export Order {$order->getEntityId()}");
        }
    }

    /**
     * @param $order
     * @param bool $saveLocal
     * @param bool $saveToFtp
     * @param bool $forced
     * @return OrderInterface
     */
    public function getPublisherData($order, bool $saveLocal = false, bool $saveToFtp = true, bool $forced = false): OrderInterface
    {
        $this->orderInterface->setOrderId($order->getEntityId());
        $this->orderInterface->setSaveLocal($saveLocal);
        $this->orderInterface->setSaveToFtp($saveToFtp);
        $this->orderInterface->setIsForced($forced);

        return $this->orderInterface;
    }
}
