<?php


namespace Drop\Pvs\Model\Consumer\Import;


use Drop\Pvs\Api\FileInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Modifier;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\File\Csv;
use Magento\Framework\Filesystem\Io\File;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\ShipmentRepositoryInterface;
use Magento\Sales\Model\Convert\Order;
use Magento\Sales\Model\Order\Email\Sender\ShipmentCommentSender;
use Magento\Sales\Model\Order\Shipment\Track;

class Shipment
{
    private $logger;
    private $configuration;
    private $utilities;
    private $modifier;
    private $csv;
    private $searchCriteriaBuilder;
    private $orderRepository;
    private $convertOrder;
    private $shipmentRepository;
    private $track;
    private $emailSender;

    /**
     * Shipment constructor.
     * @param Logger $logger
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param Modifier $modifier
     * @param Csv $csv
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param OrderRepositoryInterface $orderRepository
     * @param Order $convertOrder
     * @param ShipmentRepositoryInterface $shipmentRepository
     * @param Track $track
     * @param ShipmentCommentSender $emailSender
     */
    public function __construct(
        Logger $logger,
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        Modifier $modifier,
        Csv $csv,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        OrderRepositoryInterface $orderRepository,
        Order $convertOrder,
        ShipmentRepositoryInterface $shipmentRepository,
        Track $track,
        ShipmentCommentSender $emailSender
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->modifier = $modifier;
        $this->csv = $csv;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderRepository = $orderRepository;
        $this->convertOrder = $convertOrder;
        $this->shipmentRepository = $shipmentRepository;
        $this->track = $track;
        $this->emailSender = $emailSender;
    }

    /**
     * @param FileInterface $data
     * @throws \Exception
     */
    public function process(FileInterface $data)
    {
        if (!count($data->getFiles())) {
            return;
        }

        $carriers = $this->utilities->getCountriesCarrierArray();

        foreach ($data->getFiles() as $shipmentFile) {
            $this->csv->setEnclosure($this->configuration->getCsvEnclosure());
            $this->csv->setDelimiter($this->configuration->getCsvSeparator());
            $csvData = $this->utilities->csvToArray($this->csv->getData($shipmentFile));

            foreach ($csvData as $row) {
                try {
                    $incrementId = str_replace("-0", "", $row['order_number']);
                    $this->searchCriteriaBuilder->addFilter("increment_id", $incrementId);
                    foreach ($this->orderRepository->getList($this->searchCriteriaBuilder->create())->getItems() as $order) {
                        $shipment = $this->convertOrder->toShipment($order);
                        foreach ($order->getAllItems() as $orderItem) {
                            $qtyShipped = $orderItem->getQtyOrdered();
                            $shipmentItem = $this->convertOrder->itemToShipmentItem($orderItem)->setQty($qtyShipped);
                            $shipment->addItem($shipmentItem);
                        }
                        $shipment->register();
                        $order->setStatus($this->configuration->getShippedStatus());
                        $order->addStatusToHistory($order->getStatus(), "Create shipment");
                        $carrier = $carriers[$order->getShippingAddress()->getCountryid()] ?? $carriers['*'];
                        $this->track->addData([
                            'carrier_code' => $carrier,
                            'title'        => $carrier,
                            'number'       => $row['tracking_number'],
                            'description'  => $row['tracking_url']
                        ]);
                        $shipment->addTrack($this->track);
                        $this->shipmentRepository->save($shipment);
                        $this->orderRepository->save($order);
                        $this->emailSender->send($shipment);
                    }
                } catch (\Exception $exception) {
                    $this->logger->error($exception->getMessage());
                }
            }

            $this->manageFtpFile($shipmentFile, $data->getCanDeleteFiles());
        }
    }

    /**
     * @param $filename
     * @param $canDeleteFile
     */
    protected function manageFtpFile($filename, $canDeleteFile)
    {
        if ($canDeleteFile) {
            $this->modifier->delete($filename);
            return;
        }

        $this->modifier->moveToDone($filename);
    }
}
