<?php


namespace Drop\Pvs\Model\Consumer\Export;

use Drop\Pvs\Api\Export\RmaInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Uploader;
use Drop\Pvs\Model\Service\PdfDownloader;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Type;
use Magento\Customer\Model\Vat;
use Magento\Directory\Model\RegionFactory;
use Magento\Rma\Api\RmaRepositoryInterface;

class Rma
{
    private $configuration;
    private $utilities;
    private $rmaRepository;
    private $logger;
    private $vat;
    private $regionFactory;
    private $uploader;
    private $pdfDownloader;

    /**
     * Rma constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param UtilitiesHelper $utilitiesHelper
     * @param RmaRepositoryInterface $rmaRepository
     * @param Logger $logger
     * @param Vat $vat
     * @param RegionFactory $regionFactory
     * @param Uploader $uploader
     * @param PdfDownloader $pdfDownloader
     */
    public function __construct(
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        RmaRepositoryInterface $rmaRepository,
        Logger $logger,
        Vat $vat,
        RegionFactory $regionFactory,
        Uploader $uploader,
        PdfDownloader $pdfDownloader
    )
    {
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->rmaRepository = $rmaRepository;
        $this->logger = $logger;
        $this->vat = $vat;
        $this->regionFactory = $regionFactory;
        $this->uploader = $uploader;
        $this->pdfDownloader = $pdfDownloader;
    }

    /**
     * @param RmaInterface $data
     * @return bool
     */
    public function process(RmaInterface $data): bool
    {
        if (!$rma = $this->rmaRepository->get($data->getRmaId())) {
            $this->logger->error("rma not found [ID: {$data->getRmaId()}]");
        }

        // ordine già esportato e senza la richiesta di un export forzato
        if ($rma->getDropPvsExported() && !$data->getIsForced()) {
            return true;
        }

        $this->logger->info("Exporting Rma [ID: {$rma->getEntityId()}]");

        // genero XDOCE
        $this->logger->info("Generate XDOCE [ID: {$rma->getEntityId()}]");
        $rmaData = $this->utilities->getXmlContentFromArray($this->generateRmaData($rma));

        $this->saveFiles($rma, $rmaData, $data->getSaveToFtp(), $data->getSaveLocal());

        $rma->setDropPvsExported(1);
        $this->rmaRepository->save($rma);

        return true;
    }

    /**
     * @param $rma
     * @return array
     */
    public function generateRmaData($rma): array
    {
        try {
            $rmaData = [
                'doce' => $this->getRmaFileItems($rma)
            ];
        } catch (\Exception $e) {
            $rma->addStatusToHistory($rma->getStatus(), $e->getMessage());
            $this->rmaRepository->save($rma);
            $this->logger->error($e->getMessage());
        }

        return $rmaData;
    }

    /**
     * @param $rma
     * @return array
     */
    public function getRmaFileItems($rma): array
    {
        $items = [];
        foreach ($rma->getItems() as $item) {
            $items[] = [
                'riga' => [
                    'COD_SOCL'    => $this->configuration->getCodSocl(),
                    'COD_DICL'    => $this->configuration->getCodDicl(),
                    'COD_ART'     => $item->getProductSku(),
                    'QTA_RIGA'    => $item->getQtyRequested(),
                    'COD_UM'      => "PZ",
                    'ORDINE'      => $rma->getOrder()->getIncrementId() . '-0',
                    'COD_CLI_FOR' => $this->configuration->getCodCliFor(),
                    'BOLLA'       => $rma->getIncrementId(),
                    'DATA_BOLLA'  => $rma->getDateRequested(),
                    'COD_MCON'    => "",
                    'COD_CMOV'    => "RESO",
                ],
            ];
        }

        return $items;
    }

    /**
     * @param $rma
     * @param $rmaData
     * @param $saveToFtp
     * @param $saveLocal
     */
    public function saveFiles($rma, $rmaData, $saveToFtp, $saveLocal)
    {
        $saveOn = $saveToFtp && $saveLocal ? "Pvs FTP and Local" : ($saveToFtp ? "Pvs FTP" : "Local");
        try {
            // XDOCE
            $filename = $this->utilities->completeFilenameWithDateTime($this->configuration->getRmaFilename());
            $this->uploader->execute($filename, $rmaData, $saveToFtp, $saveLocal, "xml");
            $rma->addStatusToHistory($rma->getStatus(), "XDOCE File saved on {$saveOn}");
            $this->rmaRepository->save($rma);
            $this->logger->info("XDOCE File saved");

            $rma->addStatusToHistory($rma->getStatus(), "PDF Files saved on {$saveOn}");
            $this->logger->info("PDF Files saved");
        } catch (\Exception $e) {
            $rma->addStatusToHistory($rma->getStatus(), $e->getMessage());
            $this->logger->error($e->getMessage());
        }
        $this->rmaRepository->save($rma);
    }
}
