<?php


namespace Drop\Pvs\Model\Consumer\Export;


use Drop\Pvs\Api\Export\ProductsInterface;
use Drop\Pvs\Helper\ConfigurationHelper;
use Drop\Pvs\Helper\UtilitiesHelper;
use Drop\Pvs\Logger\Logger;
use Drop\Pvs\Model\Ftp\Uploader;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Action;
use Magento\Framework\Api\SearchCriteriaBuilder;

class Products
{
    private $logger;
    private $configuration;
    private $utilities;
    private $productsRepositoryInterface;
    private $searchCriteriaBuilder;
    private $productAction;
    private $uploader;

    /**
     * Products constructor.
     * @param Logger $logger
     * @param ConfigurationHelper $configurationHelper
     * @param ProductRepositoryInterface $productsRepositoryInterface
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Action $productAction
     * @param Uploader $uploader
     */
    public function __construct(
        Logger $logger,
        ConfigurationHelper $configurationHelper,
        UtilitiesHelper $utilitiesHelper,
        ProductRepositoryInterface $productsRepositoryInterface,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Action $productAction,
        Uploader $uploader
    )
    {
        $this->logger = $logger;
        $this->configuration = $configurationHelper;
        $this->utilities = $utilitiesHelper;
        $this->productsRepositoryInterface = $productsRepositoryInterface;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->productAction = $productAction;
        $this->uploader = $uploader;
    }

    /**
     * @param ProductsInterface $data
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function process(ProductsInterface $data)
    {
        $this->searchCriteriaBuilder->addFilter('sku', $data->getProductsIds(), 'in');
        if (!count($data->getProductsIds())) {
            return;
        }

        $productsData = $this->generateProductsData($data);
        try {
            $filename = $this->utilities->completeFilenameWithDateTime($this->configuration->getProductFilename());
            $this->uploader->execute($filename, $productsData, $data->getSaveToFtp(), $data->getSaveLocal());
            $this->logger->info("XPRODOTTI File saved");
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * @param $data
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function generateProductsData($data): array
    {
        $productsData = [];
        foreach ($data->getProductsIds() as $productId) {
            $product = $this->productsRepositoryInterface->getById($productId, true, 0, true);
            if ($product->getIsPvsExported() && !$data->getIsForced()) {
                continue;
            }

            $productsData[] = $this->generateProductData($product);
        }

        $keys = array_keys($productsData[0]);
        $productsData = array_merge([$keys], $productsData);
        return $productsData;
    }

    /**
     * @param $product
     * @return array
     */
    public function generateProductData($product): array
    {
        $productRow = [
            'COD_SOCL'    => $this->configuration->getCodSocl(),
            'COD_DICL'    => $this->configuration->getCodDicl(),
            'COD_ART'     => $product->getSku(),
            'DESCRIZIONE' => str_replace(array("\n", "\r"), "", $product->getName()),
            'COD_UM'      => 'PZ',
            'BCODE_ART'   => $product->getData($this->configuration->getEanAttributeCode()),
        ];

        $this->productAction->updateAttributes([$product->getId()], ["is_pvs_exported" => "1"], 0);
        $this->logger->info("Exported  product {$product->getSku()}");

        return $productRow;
    }
}
