<?php

namespace Drop\Pdfgenerator\Helper;

use Magento\Sales\Model\Order\Invoice;
use Magento\Sales\Model\Order\Creditmemo;
use Mirasvit\Rma\Model\Rma;
use Magento\Sales\Model\Order;

use Magento\Framework\App\Filesystem\DirectoryList;
use Mpdf\Mpdf;
use Eadesigndev\Pdfgenerator\Model\Source\TemplatePaperOrientation;

/**
 * Pdf class
 *
 * centralizza la stampa dei template pdf per Invoice, Credit memo e Rma
 */
class Pdf extends \Eadesigndev\Pdfgenerator\Helper\Pdf
{
    /**
     * @var \Magento\Sales\Model\Order\Creditmemo
     */
    public $creditmemo;

    /**
     * @var \Mirasvit\Rma\Model\Rma
     */
    public $rma;

    public $giftMessage;

    public $cleanInvoice;

    /**
     * {@inheritDoc}
     */
    public function template2Pdf()
    {
        $parts = $this->transport();

        $applySettings = $this->eaPDFSettings($parts);

        $fileParts = [
          'filestream' => $applySettings,
          'filename' => filter_var($parts['filename'], FILTER_SANITIZE_URL)
        ];

        return $fileParts;
    }

    public function setDocumentType($document)
    {
        if ($document instanceof Invoice) {
            $this->setInvoice($document);
            return $this;
        }

        if ($document instanceof Creditmemo) {
            $this->setCreditmemo($document);
            return $this;
        }

        if ($document instanceof Rma) {
            $this->setRma($document);
            return $this;
        }

        throw new \Exception("Document type not allowed!");
    }

    public function cleanInvoice() {
        $this->cleanInvoice = true;
    }

    public function setGiftMessage($message)
    {
        $this->giftMessage = $message;
    }

    /**
     * @param Creditmemo $creditmemo
     * @return $this
     */
    public function setCreditmemo(Creditmemo $creditmemo)
    {
        $this->creditmemo = $creditmemo;
        $this->setOrder($creditmemo->getOrder());
        //Magento ^2.2 bug. Will be fixed in 2.3 https://github.com/magento/magento2/issues/6849
        //$this->setInvoice($creditmemo->getInvoice());
        foreach ($creditmemo->getOrder()->getInvoiceCollection() as $invoice) {
            $this->setInvoice($invoice);
        }

        return $this;
    }

    /**
     * @param Rma $rma
     * @return $this
     */
    public function setRma(Rma $rma)
    {
        $this->rma = $rma;
        foreach ($this->getRmaManagement()->getOrder($rma)->getInvoiceCollection() as $invoice) {
            $this->setInvoice($invoice);
            break;
        }
        return $this;
    }

    public function isCee($country) {
        $euCountries = explode(',', $this->scopeConfig->getValue('general/country/eu_countries'));
        if(($country!= '') && (in_array($country, $euCountries))):
            return true;
        endif;
        return false;
    }

    /**
     * {@inheritDoc}
     */
    private function getPaymentHtml(Order $order)
    {
        return $this->paymentHelper->getInfoBlockHtml(
            $order->getPayment(),
            $order->getStoreId()
        );
    }

    /**
     * {@inheritDoc}
     */
    private function getFormattedShippingAddress(Order $order)
    {
        return $order->getIsVirtual()
            ? null
            : $this->addressRenderer->format($order->getShippingAddress(), 'html');
    }

    /**
     * {@inheritDoc}
     */
    private function getFormattedBillingAddress(Order $order)
    {
        /** @var \Magento\Sales\Model\Order\Address $billing */
        $billing = $order->getBillingAddress();
        $address = $this->addressRenderer->format($billing, 'html');
        return $address;
    }

    /**
     * {@inheritDoc}
     */
    private function transport()
    {
        $invoice = $this->invoice;
        $order = $this->order;
        $paymentTitle= $order->getPayment()->getMethodInstance()->getTitle();

        $invoice = '';
        $invoice_comment = '';
        if ($this->invoice instanceof Invoice) {
            $invoice = $this->invoice;
            $invoice_comment = $invoice->getCustomerNoteNotify() ? $invoice->getCustomerNote() : '';
        }

        $creditmemo = '';
        $creditmemo_comment = '';
        if ($this->creditmemo instanceof Creditmemo) {
            $creditmemo = $this->creditmemo;
            $creditmemo_comment = $creditmemo->getCustomerNoteNotify() ? $creditmemo->getCustomerNote() : '';
        }

        $rma = '';
        $rmaItemHtml = '';
        $rmaTotal = 0;
        if ($this->rma instanceof Rma) {
            $rma = $this->rma;
            $items = $this->getRmaSearchManagement()->getRequestedItems($rma);
            foreach ($items as $item) {
                $itemOrderId = $this->getItemManagement()->getOrderItem($item)->getOrderId();
                if ($itemOrderId == $order->getId()) {
                    $mageItem = $this->getOrderItemRepository()->get($item->getOrderItemId());

                    $calculatedItemQty = $item->getQtyRequested();
                    $calculatedPrice = ($mageItem->getPrice() - (($mageItem->getDiscountAmount() * 100) / ($mageItem->getTaxPercent() + 100)) + $mageItem->getTaxAmount()) * $calculatedItemQty;

                    # fix for get price from bundle product
                    $productOptions = $mageItem->getProductOptions();
                    if (isset($productOptions['bundle_selection_attributes'])) {
                        $bundleOptions = json_decode($productOptions['bundle_selection_attributes'], true);

                        $calculatedItemQty = $bundleOptions['qty'];
                        $calculatedPrice = ($bundleOptions['qty'] * $bundleOptions['price']);
                    }

                    $rmaItemHtml .= "
                        <tr>
                            <td>
                                <p class='product-name'>".$mageItem->getName()."</p>
                                <p class='sku'>SKU: ".$mageItem->getSku()."</p>
                            </td>
                            <td>" . $calculatedItemQty . "</td>
                            <td>" . $order->formatPrice($calculatedPrice) . "</td>
                            <td><span>" . __($this->getReasonRepository()->get($item->getReasonId())->getName()) ."</span></td>
                        </tr>";
                    $rmaTotal += $calculatedPrice;
                }
            }
        }

        $proformaListHtml = $productsListHtml = '';
        foreach ($order->getAllVisibleItems() as $item) {
            $itemQty = $item->getQtyOrdered();
            $itemPrice = ($item->getPrice() - (($item->getDiscountAmount() * 100) / ($item->getTaxPercent() + 100)) + $item->getTaxAmount()) * $itemQty;

            $productsListHtml .=  "
                        <tr>
                            <td>
                                <p class='product-name'>".$item->getName()."</p>
                                <p class='sku'>SKU: ".$item->getSku()."</p>
                            </td>
                            <td>" . $item->getQtyOrdered() . "</td>
                        </tr>";

            $proformaListHtml .=  "
                        <tr>
                            <td><div class='square'>&nbsp;</div></td>
                            <td>
                                <p class='product-name'>".$item->getName()."</p>
                                <p class='sku'>SKU: ".$item->getSku()."</p>
                            </td>
                            <td>" . $itemQty . "</td>
                            <td>" . $order->formatPrice($itemPrice) . "</td>
                        </tr>";
        }

        $writer = new \Zend\Log\Writer\Stream(BP . '/var/log/pvs.log');
        $logger = new \Zend\Log\Logger();
        $logger->addWriter($writer);
        $logger->debug("[PVS][SAVE_DOC_FILE_FTP][PDF] invoice #{$invoice->getIncrementId()} order #{$order->getIncrementId()} - pdf");

        $transport = [
            'menaged_drop_address' => $order->getIsVirtual() ? "" : "Managed by: " . $this->getFatturazioneElettronicaUtilities()->getManagedDropAddress($order),
            'rea_drop_info' => $order->getIsVirtual() ? "" : $this->getFatturazioneElettronicaUtilities()->getReaDropInfo($order),
            'order' => $order,
            'invoice' => $invoice,
            'creditmemo' => $creditmemo,
            'rma' => $rma,
            'rma_items' => $rmaItemHtml,
            'rma_total' => $order->formatPrice($rmaTotal),
            'products_list' => $productsListHtml,
            'proforma_items' => $proformaListHtml,
            'invoice_comment' => $invoice_comment,
            'creditmemo_comment' => $creditmemo_comment,
            'billing' => $order->getBillingAddress(),
            'payment_html' => $this->getPaymentHtml($order),
            'paymentInfo' => $order->getPayment()->getMethodInstance()->getTitle(),
            'store' => $order->getStore(),
            'formattedShippingAddress' => $this->getFormattedShippingAddress($order),
            'formattedBillingAddress' => $this->getFormattedBillingAddress($order),
            'logo_path' => $this->getFileSystem()->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath() . 'email/logo/' . $this->getHelper()->hasConfig('design/email/logo'),
            'return_address' => nl2br($this->getHelper()->hasConfig('rma/general/return_address')),
            'media_path' => $this->getFileSystem()->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath(),
            'grand_total_no_gift' => $order->formatPrice($order->getGrandTotal() + $order->getGiftCardsAmount())
        ];

        if($order->getGiftMessageId() && $this->giftMessage) {
            $transport['gift_message'] = $this->giftMessage;
        }

        if(!($this->rma instanceof Rma)) {
            $country_id = $order->getBillingAddress()->getCountryId();

            if (!$this->isCee($country_id) && !($this->cleanInvoice)) {
                $transport['add_washington'] = true;
            }

            if($country_id == 'IT') {
                $transport['it_customer'] = true;
            }
            else {
                $this->isCee($country_id) ? $transport['eu_customer'] = true : $transport['ee_customer'] = true;
            }
        }

        $processor = $this->processor;
        $processor->setVariables($transport);
        $processor->setTemplate($this->template);
        $parts = $processor->processTemplate();

        return $parts;
    }

    /**
     * {@inheritDoc}
     */
    private function eaPDFSettings($parts)
    {
        $templateModel = $this->template;

        if (!$templateModel->getTemplateCustomForm()) {
            /** @var mPDF $pdf */
            //@codingStandardsIgnoreLine
            $pdf = new Mpdf($this->config($templateModel));
        }

        if ($templateModel->getTemplateCustomForm()) {
            //@codingStandardsIgnoreLine
            $pdf = new Mpdf($this->config($templateModel));
        }

        $pdf->SetHTMLHeader($parts['header']);
        $pdf->SetHTMLFooter($parts['footer']);

        $pdf->WriteHTML($templateModel->getTemplateCss(), 1);

        //@codingStandardsIgnoreLine
        $pdf->WriteHTML('<body>' . html_entity_decode($parts['body']) . '</body>');
        $pdfToOutput = $pdf->Output('', 'S');


        return $pdfToOutput;
    }

    /**
     * @param Pdfgenerator $templateModel
     * @return array
     */
    private function config($templateModel)
    {
        $ori = $templateModel->getTemplatePaperOri() ?? 1;
        $orientation = $this->getTemplatePaperOrientation()->getAvailable();
        $finalOri = isset($orientation[$ori][0]) ? $orientation[$ori][0] : 'P';
        $marginTop = $templateModel->getTemplateCustomT();
        $marginBottom = $templateModel->getTemplateCustomB();
        $paperForms = $this->getTemplatePaperForm()->getAvailable();
        $templatePaperForm = $templateModel->getTemplatePaperForm();
        if (!$templatePaperForm) {
            $templatePaperForm = 1;
        }
        $form = $paperForms[$templatePaperForm];
        if ($ori == TemplatePaperOrientation::TEMAPLATE_PAPER_LANDSCAPE) {
            $form = $paperForms[$templateModel->getTemplatePaperForm()] . '-' . $finalOri;
        }

        $config = [
          'mode' => 'utf-8',
          'format' => $form,
          'setAutoTopMargin' => 'pad',
          'setAutoBottomMargin' => 'pad',
          'autoMarginPadding' => 0,
          'default_font_size' => '',
          'default_font' => '',
          'margin_left' => $templateModel->getTemplateCustomL(),
          'margin_right' => $templateModel->getTemplateCustomR(),
          'margin_top' => 0,
          'margin_bottom' => 0,
          'margin_header' => $marginTop,
          'margin_footer' => $marginBottom,
          'tempDir' => $this->getDirectoryList()->getPath('tmp')
        ];

        if ($templateModel->getTemplateCustomForm()) {
            $config = [
              'mode' => 'utf-8',
              'format' => [
                  $templateModel->getTemplateCustomW(),
                  $templateModel->getTemplateCustomH()
              ],
              'setAutoTopMargin' => 'pad',
              'setAutoBottomMargin' => 'pad',
              'autoMarginPadding' => 0,
              'default_font_size' => '',
              'default_font' => '',
              'margin_left' => $templateModel->getTemplateCustomL(),
              'margin_right' => $templateModel->getTemplateCustomR(),
              'margin_top' => 0,
              'margin_bottom' => 0,
              'margin_header' => $marginTop,
              'margin_footer' => $marginBottom,
              'tempDir' => $this->getDirectoryList()->getPath('tmp')
          ];
        }

        return $config;
    }

    private function getReasonRepository()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
          ->get(\Mirasvit\Rma\Api\Repository\ReasonRepositoryInterface::class);
    }

    private function getRmaSearchManagement()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
      ->get(\Mirasvit\Rma\Api\Service\Rma\RmaManagement\SearchInterface::class);
    }

    private function getItemManagement()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
      ->get(\Mirasvit\Rma\Api\Service\Item\ItemManagementInterface::class);
    }

    private function getOrderItemRepository()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
      ->get(\Magento\Sales\Model\Order\ItemRepository::class);
    }

    private function getFilesystem()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Magento\Framework\Filesystem::class);
    }


    private function getHelper()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Eadesigndev\Pdfgenerator\Helper\Data::class);
    }

    private function getTemplatePaperOrientation()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Eadesigndev\Pdfgenerator\Model\Source\TemplatePaperOrientation::class);
    }

    private function getTemplatePaperForm()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Eadesigndev\Pdfgenerator\Model\Source\TemplatePaperForm::class);
    }

    private function getDirectoryList()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Magento\Framework\App\Filesystem\DirectoryList::class);
    }

    private function getRmaManagement()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Mirasvit\Rma\Service\Rma\RmaManagement::class);
    }

    private function getFatturazioneElettronicaUtilities()
    {
        return \Magento\Framework\App\ObjectManager::getInstance()
      ->get(\Drop\FatturazioneElettronica\Model\Utilities::class);
    }
}
