<?php


namespace Drop\Oms\Plugin\Magento\Rma;


use Drop\Oms\Helper\ConfigurationHelper;
use Psr\Log\LoggerInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Rma\Helper\Data;
use Magento\Sales\Api\OrderRepositoryInterface;

class DataHelper
{
    protected $configuration;
    protected $orderRepository;
    protected $logger;

    /**
     * RmaDataHelper constructor.
     * @param ConfigurationHelper $configurationHelper
     * @param OrderRepositoryInterface $orderRepository
     * @param LoggerInterface $logger
     */
    public function __construct(
        ConfigurationHelper      $configurationHelper,
        OrderRepositoryInterface $orderRepository,
        LoggerInterface          $logger
    )
    {
        $this->configuration = $configurationHelper;
        $this->orderRepository = $orderRepository;
        $this->logger = $logger;
    }

    /**
     * @param Data $subject
     * @param $result
     * @param $order
     * @return false|mixed|void
     * @throws LocalizedException
     */
    public function afterCanCreateRma(Data $subject, $result, $order)
    {
        // se non è attivo il controllo dei giorni
        // se magento nega la possibilità
        // ==> torno il risultato di magento
        if (
            !$this->configuration->isReturnsDayLimitEnabled()
            || !$result
        ) {
            return $result;
        }

        if (!is_object($order)) {
            $order = $this->orderRepository->get($order);
        }

        if (!$order) {
            $this->logger->error("[DROP][OMS][CAN CREATE RMA] error, can't find order");
            throw new LocalizedException(__('Please check this order for errors.'));
            return;
        }

        if (!$order->getDropPvsDeliveryDate()) {
            return false;
        }

        return $this->checkDateDifference($order->getDropPvsDeliveryDate(), $result);
    }

    public function checkDateDifference($deliveryDate, $result)
    {
        $difference = $this->getWorkingDays($deliveryDate);

        if ($difference > (int)$this->configuration->getReturnsDayLimit()) {
            return false;
        }

        return $result;
    }

    /**
     * @param $from
     * @param $to
     * @return int
     * @throws \Exception
     */
    public function getWorkingDays($from, $to = null)
    {
        $workingDays = [1, 2, 3, 4, 5];
        $holidayDays = ['*-01-01', '*-01-06', '*-04-25', '*-05-01', '*-06-02', '*-08-15', '*-11-01', '*-12-08', '*-12-25', '*-12-26'];
        // manca la Pasqua che è un giorno variabile...
        $from = new \DateTime($from);
        $to = $to ? new \DateTime($to) : new \DateTime(date('Y-m-d'));
        $to->modify('+1 day');
        $interval = new \DateInterval('P1D');
        $periods = new \DatePeriod($from, $interval, $to);

        $days = 0;
        foreach ($periods as $period) {
            if (!in_array($period->format('N'), $workingDays)) continue;
            if (in_array($period->format('Y-m-d'), $holidayDays)) continue;
            if (in_array($period->format('*-m-d'), $holidayDays)) continue;
            $days++;
        }
        return $days;
    }
}
