<?php

namespace Drop\Oms\Plugin\Magento\Quote\Model;

use Drop\Oms\Block\Checkout\Address\Fields\ShippingLayoutProcessor;
use Magento\Framework\Exception\CouldNotSaveException;

class ShippingAddressManagementPlugin
{
    /**
     * @var \Magento\Framework\App\ProductMetadataInterface
     */
    private $productMetadata;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var
     */
    protected $version;

    /**
     * ShippingAddressManagementPlugin constructor.
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        \Magento\Framework\App\ProductMetadataInterface $productMetadata,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    )
    {
        $this->productMetadata = $productMetadata;
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * @param $subject
     * @param $cartId
     * @param $address
     * @throws CouldNotSaveException
     */
    public function beforeAssign(
        $subject,
        $cartId,
        $address
    )
    {
        $this->version = $this->productMetadata->getVersion();

        $extAttributes = $this->getExtensionAttributesArray($address);

        if ($extAttributes) {

            //Force request invoice if current country is in $mandatoryRequestInvoiceCountries array
            $mandatoryRequestInvoiceCountries = explode(',', $this->scopeConfig->getValue('oms/general/req_request_invoice_countries'));
            if (in_array($address->getCountryId(), $mandatoryRequestInvoiceCountries)) {
                if ($this->version < '2.4') {
                    $extAttributes->setRequestInvoice(1);
                }
                $address->setRequestInvoice(1);
            }

            try {
                $requestInvoice = $this->getFiled($extAttributes, ShippingLayoutProcessor::REQUEST_INVOICE);
                $address->setRequestInvoice($requestInvoice);

                $customerTypology = 0;
                $fiscalCode = '';
                $receptionDestination = '';

                if ($requestInvoice) {
                    $customerTypology = $this->getFiled($extAttributes, ShippingLayoutProcessor::CUSTOMER_TYPOLOGY);
                    $receptionDestination = $this->getFiled($extAttributes, ShippingLayoutProcessor::RECEPTION_DESTINATION);

                    if ($customerTypology == 0) {
                        //Reset entered data based on customer typology
                        $fiscalCode = $this->getFiled($extAttributes, ShippingLayoutProcessor::FISCAL_CODE);
                        $address->setVatId('');
                        $address->setCompany('');
                    }
                }

                $address->setCustomerTypology($customerTypology);
                $address->setFiscalCode($fiscalCode);
                $address->setFeReceptionDestination($receptionDestination);

                $this->forceData($address);

            } catch (\Exception $e) {
                throw new CouldNotSaveException(
                    __('One custom field could not be added to the address.'),
                    $e
                );
            }
        }
    }

    /**
     * @param $address
     * @return array
     */
    protected function getExtensionAttributesArray($address)
    {
        if ($this->version >= '2.4') {
            return [
                ShippingLayoutProcessor::REQUEST_INVOICE       => is_array($address->getRequestInvoice()) ? $address->getRequestInvoice()['value'] : $address->getRequestInvoice(),
                ShippingLayoutProcessor::CUSTOMER_TYPOLOGY     => is_array($address->getCustomerTypology()) ? $address->getCustomerTypology()['value'] : $address->getCustomerTypology(),
                ShippingLayoutProcessor::FISCAL_CODE           => is_array($address->getFiscalCode()) ? $address->getFiscalCode()['value'] : $address->getFiscalCode(),
                ShippingLayoutProcessor::RECEPTION_DESTINATION => is_array($address->getFeReceptionDestination()) ? $address->getFeReceptionDestination()['value'] : $address->getFeReceptionDestination(),
            ];
        }

        return $address->getExtensionAttributes();
    }

    /**
     * @param $extensionAttributes
     * @param $field
     * @return mixed
     */
    protected function getFiled($extensionAttributes, $field)
    {
        if ($this->version >= '2.4') {
            return $extensionAttributes[$field];
        }

        switch ($field) {
            case ShippingLayoutProcessor::REQUEST_INVOICE:
                return $extensionAttributes->getRequestInvoice();
            case ShippingLayoutProcessor::CUSTOMER_TYPOLOGY:
                return $extensionAttributes->getCustomerTypology();
            case ShippingLayoutProcessor::FISCAL_CODE:
                return $extensionAttributes->getFiscalCode();
            case ShippingLayoutProcessor::RECEPTION_DESTINATION:
                return $extensionAttributes->getFeReceptionDestination();
        }
    }

    /**
     * @param $address
     */
    protected function forceData($address)
    {
        if ($this->version < '2.4') {
            return;
        }

        $address->setData($address->getData());
        $address->setRequestInvoice($this->clearData(ShippingLayoutProcessor::REQUEST_INVOICE, $address));
        $address->setCustomerTypology($this->clearData(ShippingLayoutProcessor::CUSTOMER_TYPOLOGY, $address));
        $address->setFiscalCode($this->clearData(ShippingLayoutProcessor::FISCAL_CODE, $address));
        if (strpos($address->getData(ShippingLayoutProcessor::RECEPTION_DESTINATION), "@") === false) {
            $address->setFeReceptionDestination($this->clearData(ShippingLayoutProcessor::RECEPTION_DESTINATION, $address));
        } else {
            $address->setFeReceptionDestination($this->clearData(ShippingLayoutProcessor::RECEPTION_DESTINATION, $address, true));
        }

        $address->setData($address->getData());
    }

    /**
     * @param $field
     * @param $address
     * @param false $noPreg
     * @return array|string|string[]|null
     */
    protected function clearData($field, $address, $noPreg = false)
    {
        if ($noPreg) {
            return trim(str_replace($field, '', $address->getData($field)));
        }
        return preg_replace('/[^A-Za-z0-9\-]/', '', str_replace($field, '', $address->getData($field)));
    }
}
