<?php

namespace Drop\Oms\Block\Address\Edit;

use Magento\Framework\View\Element\Template;
use \Drop\Oms\Block\Checkout\Address\Fields\ShippingLayoutProcessor;
use \Drop\Oms\Helper\Oms as OmsHelper;

class Oms extends Template
{
    /**
     * @var string
     */
    protected $_template = 'address/edit/field/oms_fields.phtml';

    /**
     * @var \Magento\Customer\Api\Data\AddressInterface
     */
    protected $_address;

    /**
     * @var \Magento\Framework\View\Element\Template\File\Validator
     */
    protected $scopeConfig;
    /**
     * @var OmsHelper
     */
    protected $helper;
    /**
     * @var \Magento\Directory\Helper\Data
     */
    protected $directoryHelper;

    /**
     * Oms constructor.
     * @param Template\Context $context
     * @param OmsHelper $helper
     * @param \Magento\Directory\Helper\Data $directoryHelper
     * @param array $data
     */
    public function __construct(
        Template\Context $context,
        OmsHelper $helper,
        \Magento\Directory\Helper\Data $directoryHelper,
        array $data = []
    )
    {
        parent::__construct($context, $data);
        $this->scopeConfig = $context->getScopeConfig();
        $this->helper = $helper;
        $this->directoryHelper = $directoryHelper;
    }

    /**
     * @return bool
     */
    public function forceRequestInvoice()
    {
        return $this->helper->forceRequestInvoice($this->getCountryId());
    }

    /**
     * @return bool
     */
    public function canBePrivateCustomer()
    {
        return $this->helper->canBePrivateCustomer($this->getCountryId());
    }

    /**
     * @return string
     */
    public function getRequestInvoice()
    {
        /** @var \Magento\Customer\Model\Data\Address $address */
        $address = $this->getAddress();

        $requestInvoice = $address->getCustomAttribute(ShippingLayoutProcessor::REQUEST_INVOICE);

        if (!$requestInvoice) {
            return false;
        }

        if ($requestInvoice->getValue() == 1) {
            return true;
        }
        return false;
    }

    public function checkRequestInvoice()
    {
        if ($this->getRequestInvoice()) {
            return "checked='checked'";
        }
        return '';
    }

    public function isMandatoryRequestInvoice()
    {
        $address = $this->getAddress();
        $countryId = $address->getCountryId();
        $requestInvoiceCountries = $this->getRequestInvoiceCountriesConfiguration();
        if (!empty($countryId) && !empty($requestInvoiceCountries) && in_array($countryId, $requestInvoiceCountries)) {
            return true;
        }
        return false;
    }

    public function getRequestInvoiceCountriesConfiguration()
    {
        return explode(',', $this->scopeConfig->getValue('oms/general/req_request_invoice_countries'));
    }

    /**
     * @return string
     */
    public function getCustomerTypology($customerTipologyRadioValue)
    {
        /** @var \Magento\Customer\Model\Data\Address $address */
        $address = $this->getAddress();
        $customerTypology = $address->getCustomAttribute(ShippingLayoutProcessor::CUSTOMER_TYPOLOGY);
        if (!$customerTypology) {
            return '';
        }
        if (($customerTypology->getValue() == $customerTipologyRadioValue)) {
            return "checked='checked'";
        }
        return '';
    }

    /**
     * @return string
     */
    public function getFiscalCode()
    {
        /** @var \Magento\Customer\Model\Data\Address $address */
        $address = $this->getAddress();
        $fiscalCode = $address->getCustomAttribute(ShippingLayoutProcessor::FISCAL_CODE);
        if (!$fiscalCode) {
            return '';
        }
        return $fiscalCode->getValue();
    }

    /**
     * @return string
     */
    public function getReceptionDestination()
    {
        /** @var \Magento\Customer\Model\Data\Address $address */
        $address = $this->getAddress();
        $receptionDestination = $address->getCustomAttribute(ShippingLayoutProcessor::RECEPTION_DESTINATION);
        if (!$receptionDestination) {
            return '';
        }
        return $receptionDestination->getValue();
    }

    public function getDebugConfiguration()
    {
        return $this->scopeConfig->getValue('oms/general/debug');
    }

    /**
     * Return the associated address.
     *
     * @return \Magento\Customer\Api\Data\AddressInterface
     */
    public function getAddress()
    {
        return $this->_address;
    }

    /**
     * @return string|null
     */
    public function getCountryId()
    {
        $address = $this->getAddress();
        if (!$countryId = $address->getCountryId()) {
            $countryId = $this->directoryHelper->getDefaultCountry();
        }

        return $countryId;
    }

    /**
     * Set the associated address.
     *
     * @param \Magento\Customer\Api\Data\AddressInterface $address
     */
    public function setAddress(\Magento\Customer\Api\Data\AddressInterface $address)
    {
        $this->_address = $address;
    }

    public function getVatIdValidationIsEnabled()
    {
        return $this->scopeConfig->getValue('oms/general/vatid_ws_validation');
    }

    public function getVatIdValidationAjaxUrl()
    {
        return $this->getUrl('mfe/validation/vatid', ['_secure' => true]);
    }

    public function getFiscalCodeValidationIsEnabled()
    {
        return $this->scopeConfig->getValue('oms/general/fiscalcode_validation');
    }

    public function getFiscalCodeValidationAjaxUrl()
    {
        return $this->getUrl('mfe/validation/fiscalcode', ['_secure' => true]);
    }

    public function getForceRequestInvoiceAjaxUrl()
    {
        return $this->getUrl('mfe/check/requestinvoice', ['_secure' => true]);
    }

    public function getForceCanBePrivateCustomerAjaxUrl()
    {
        return $this->getUrl('mfe/check/privatecustomer', ['_secure' => true]);
    }
}
